
/*

LinkR.cpp

Receive objects via ZeriLink

Version : 2.60
Date    : 18 June 2006
Author  : John Kortink, (c) Zeridajh 1993..2006

History

<2.00
-----
- Early versions in various states, in Turbo Pascal, for different sets of hardware.
  These date back to May 1993, when the PC <--> Acorn hardware link project started.

2.00
----
- Full rewrite in C. First version for ECP hardware.

2.10
----
- Changed ECP initialisation values.
- Changed cable wiring.
- Fixed bug in %age calculations for very large files.
- Added option of running diagnostics.

2.20
----
- Changed ECP initialisation values.
- Changed cable wiring.

2.30
----
- Improved low level code : transfer speed roughly doubled.

2.40
----
- Added 'bidirectional only' transfer mode.

2.41
----
- Made code Freeware instead of Shareware.

2.42
----
- Added 'Info' object (internal use only).
- Added 'NoSynchronise' keyword (internal use only).
- Ensured succesful compilation (of asm statements in particular) on GCC 3.1.
- Usage message mistakenly mentioned <filespec>... at end of syntax.

2.43
----
- A few cosmetic changes.

2.44
----
- Internal maintenance release.

2.50
----
- Fixed asm's with backslash newlines for GCC 4.0.1.
- Internal cleanup, tweaks, recompile and rerelease.
- Changes to file date transmission.
- Ported to Unix (now RISC OS/Windows/Unix compatible).

2.60
----
- Very minor internal changes.

*/

#include <map>
#include <string>
#include <fstream>
#include <iostream>
#include <utime.h>
#include "stats.h"
#include "lib_make_string.h"
#include "lib_neat_message.h"

using namespace std;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Non-code
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#define CODE_NAME	"LinkR"
#define CODE_VERSION	"2.60 (18 June 2006)"
#define CODE_COPYRIGHT	"Copyright (c) Zeridajh 1993..2006"

#define LINK_ASM

#undef DEBUG

#define OBJECT_NULL		0
#define OBJECT_ACORNFILE	1
#define OBJECT_WINDOWSFILE	2
#define OBJECT_UNIXFILE		3
#define OBJECT_INFO		99

#ifdef UNIX

#include <glob.h>
#include <sys/io.h>

#define inportb(a)		inb(a)
#define outportb(a,b)		outb(b,a)
#define CANONICALISE(a,b)	realpath(a,b)

#endif

#ifdef WINDOWS

#include <dos.h>
#include <sys/stat.h>
#define CANONICALISE(a,b)	_fixpath(a,b)

#endif

#define SYS_SLASH	'/'

#define MODE_ECP	1
#define MODE_BIDIR	2

#define PPORT_OFF_DATA	0x000
#define PPORT_OFF_DSR	0x001
#define PPORT_OFF_DCR	0x002
#define PPORT_OFF_FIFO	0x400
#define PPORT_OFF_ECR	0x402

#define SMALL_BUFFER_SIZE	1024
#define LARGE_BUFFER_SIZE	65536

#define MAXIMUM_PATH_SIZE	1024

typedef unsigned long	nval32;
typedef unsigned short	nval16;
typedef unsigned char	nval8;
typedef signed long	zval32;
typedef signed short	zval16;
typedef signed char	zval8;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Globals
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

nval16	Link_ECR;
nval16	Link_DCR;
nval16	Link_DSR;
nval16	Link_DATA;
nval16	Link_FIFO;

nval8	Config_Mode;
nval16	Config_LinkPort;
nval8	Config_MapChar[256];
bool	Config_NoFileAccess;
bool	Config_NoSynchronise;
nval32	Config_LeafLengthExt;
nval32	Config_LeafLengthName;

map<nval32,string>	Input_Typemaps;

bool	Be_Verbose = true;
nval32	CRC32_Value;
nval8*	Buffer_Start;
nval8*	Buffer_Walker;
nval32	ECP_FIFO_Size;
nval8*	Small_Data_Buffer;
nval8*	Large_Data_Buffer;

Statistics	The_Stats;

Make_String	make_string;

nval32		CRC32_Table[] =
		{
			0x00000000ul, 0x77073096ul, 0xEE0E612Cul, 0x990951BAul, 0x076DC419ul, 0x706AF48Ful, 0xE963A535ul, 0x9E6495A3ul,
			0x0EDB8832ul, 0x79DCB8A4ul, 0xE0D5E91Eul, 0x97D2D988ul, 0x09B64C2Bul, 0x7EB17CBDul, 0xE7B82D07ul, 0x90BF1D91ul,
			0x1DB71064ul, 0x6AB020F2ul, 0xF3B97148ul, 0x84BE41DEul, 0x1ADAD47Dul, 0x6DDDE4EBul, 0xF4D4B551ul, 0x83D385C7ul,
			0x136C9856ul, 0x646BA8C0ul, 0xFD62F97Aul, 0x8A65C9ECul, 0x14015C4Ful, 0x63066CD9ul, 0xFA0F3D63ul, 0x8D080DF5ul,
			0x3B6E20C8ul, 0x4C69105Eul, 0xD56041E4ul, 0xA2677172ul, 0x3C03E4D1ul, 0x4B04D447ul, 0xD20D85FDul, 0xA50AB56Bul,
			0x35B5A8FAul, 0x42B2986Cul, 0xDBBBC9D6ul, 0xACBCF940ul, 0x32D86CE3ul, 0x45DF5C75ul, 0xDCD60DCFul, 0xABD13D59ul,
			0x26D930ACul, 0x51DE003Aul, 0xC8D75180ul, 0xBFD06116ul, 0x21B4F4B5ul, 0x56B3C423ul, 0xCFBA9599ul, 0xB8BDA50Ful,
			0x2802B89Eul, 0x5F058808ul, 0xC60CD9B2ul, 0xB10BE924ul, 0x2F6F7C87ul, 0x58684C11ul, 0xC1611DABul, 0xB6662D3Dul,
			0x76DC4190ul, 0x01DB7106ul, 0x98D220BCul, 0xEFD5102Aul, 0x71B18589ul, 0x06B6B51Ful, 0x9FBFE4A5ul, 0xE8B8D433ul,
			0x7807C9A2ul, 0x0F00F934ul, 0x9609A88Eul, 0xE10E9818ul, 0x7F6A0DBBul, 0x086D3D2Dul, 0x91646C97ul, 0xE6635C01ul,
			0x6B6B51F4ul, 0x1C6C6162ul, 0x856530D8ul, 0xF262004Eul, 0x6C0695EDul, 0x1B01A57Bul, 0x8208F4C1ul, 0xF50FC457ul,
			0x65B0D9C6ul, 0x12B7E950ul, 0x8BBEB8EAul, 0xFCB9887Cul, 0x62DD1DDFul, 0x15DA2D49ul, 0x8CD37CF3ul, 0xFBD44C65ul,
			0x4DB26158ul, 0x3AB551CEul, 0xA3BC0074ul, 0xD4BB30E2ul, 0x4ADFA541ul, 0x3DD895D7ul, 0xA4D1C46Dul, 0xD3D6F4FBul,
			0x4369E96Aul, 0x346ED9FCul, 0xAD678846ul, 0xDA60B8D0ul, 0x44042D73ul, 0x33031DE5ul, 0xAA0A4C5Ful, 0xDD0D7CC9ul,
			0x5005713Cul, 0x270241AAul, 0xBE0B1010ul, 0xC90C2086ul, 0x5768B525ul, 0x206F85B3ul, 0xB966D409ul, 0xCE61E49Ful,
			0x5EDEF90Eul, 0x29D9C998ul, 0xB0D09822ul, 0xC7D7A8B4ul, 0x59B33D17ul, 0x2EB40D81ul, 0xB7BD5C3Bul, 0xC0BA6CADul,
			0xEDB88320ul, 0x9ABFB3B6ul, 0x03B6E20Cul, 0x74B1D29Aul, 0xEAD54739ul, 0x9DD277AFul, 0x04DB2615ul, 0x73DC1683ul,
			0xE3630B12ul, 0x94643B84ul, 0x0D6D6A3Eul, 0x7A6A5AA8ul, 0xE40ECF0Bul, 0x9309FF9Dul, 0x0A00AE27ul, 0x7D079EB1ul,
			0xF00F9344ul, 0x8708A3D2ul, 0x1E01F268ul, 0x6906C2FEul, 0xF762575Dul, 0x806567CBul, 0x196C3671ul, 0x6E6B06E7ul,
			0xFED41B76ul, 0x89D32BE0ul, 0x10DA7A5Aul, 0x67DD4ACCul, 0xF9B9DF6Ful, 0x8EBEEFF9ul, 0x17B7BE43ul, 0x60B08ED5ul,
			0xD6D6A3E8ul, 0xA1D1937Eul, 0x38D8C2C4ul, 0x4FDFF252ul, 0xD1BB67F1ul, 0xA6BC5767ul, 0x3FB506DDul, 0x48B2364Bul,
			0xD80D2BDAul, 0xAF0A1B4Cul, 0x36034AF6ul, 0x41047A60ul, 0xDF60EFC3ul, 0xA867DF55ul, 0x316E8EEFul, 0x4669BE79ul,
			0xCB61B38Cul, 0xBC66831Aul, 0x256FD2A0ul, 0x5268E236ul, 0xCC0C7795ul, 0xBB0B4703ul, 0x220216B9ul, 0x5505262Ful,
			0xC5BA3BBEul, 0xB2BD0B28ul, 0x2BB45A92ul, 0x5CB36A04ul, 0xC2D7FFA7ul, 0xB5D0CF31ul, 0x2CD99E8Bul, 0x5BDEAE1Dul,
			0x9B64C2B0ul, 0xEC63F226ul, 0x756AA39Cul, 0x026D930Aul, 0x9C0906A9ul, 0xEB0E363Ful, 0x72076785ul, 0x05005713ul,
			0x95BF4A82ul, 0xE2B87A14ul, 0x7BB12BAEul, 0x0CB61B38ul, 0x92D28E9Bul, 0xE5D5BE0Dul, 0x7CDCEFB7ul, 0x0BDBDF21ul,
			0x86D3D2D4ul, 0xF1D4E242ul, 0x68DDB3F8ul, 0x1FDA836Eul, 0x81BE16CDul, 0xF6B9265Bul, 0x6FB077E1ul, 0x18B74777ul,
			0x88085AE6ul, 0xFF0F6A70ul, 0x66063BCAul, 0x11010B5Cul, 0x8F659EFFul, 0xF862AE69ul, 0x616BFFD3ul, 0x166CCF45ul,
			0xA00AE278ul, 0xD70DD2EEul, 0x4E048354ul, 0x3903B3C2ul, 0xA7672661ul, 0xD06016F7ul, 0x4969474Dul, 0x3E6E77DBul,
			0xAED16A4Aul, 0xD9D65ADCul, 0x40DF0B66ul, 0x37D83BF0ul, 0xA9BCAE53ul, 0xDEBB9EC5ul, 0x47B2CF7Ful, 0x30B5FFE9ul,
			0xBDBDF21Cul, 0xCABAC28Aul, 0x53B39330ul, 0x24B4A3A6ul, 0xBAD03605ul, 0xCDD70693ul, 0x54DE5729ul, 0x23D967BFul,
			0xB3667A2Eul, 0xC4614AB8ul, 0x5D681B02ul, 0x2A6F2B94ul, 0xB40BBE37ul, 0xC30C8EA1ul, 0x5A05DF1Bul, 0x2D02EF8Dul
		};

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Messages
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void Issue_Fatal_Error(string message_text)
{
	cerr << Neat_Message(CODE_NAME ": ", "fatal error: " + message_text + "\n");

	exit(1);
}

void Issue_Usage_Error(string message_text)
{
	cerr << Neat_Message(CODE_NAME ": ", "usage error: " + message_text + "\n");

	cerr <<

	"Usage: " CODE_NAME " [option...]\n"
	"\n"
	"[option] is one of :\n"
	"-d : perform diagnostics only\n"
	"-q : be quiet (defeats -v)\n"
	"-v : be verbose (default, defeats -q)\n"
	"\n"
	"[x] means x is optional\n"
	"<x> means x is required\n"
	"x... means one or more times x\n"

	;

	exit(1);
}

void Issue_Fatal_InLine(string file_path, int line_number, string message_text)
{
	cerr << Neat_Message(CODE_NAME ": ", "error in line " + MAKE_STRING(line_number) + " of '" + file_path + "' :");

	Issue_Fatal_Error(message_text);
}

void Issue_Fatal_OpenFile(string file_path)
{
	Issue_Fatal_Error("cannot open file '" + file_path + "'");
}

#ifdef DEBUG

void log(string message)
{
	ofstream out_file("e:\\zllog.txt", ofstream::out | ofstream::app);
	out_file << "Message : '" << message << "'\n";
}

#endif

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Main
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void Link_Diagnostics_ECP(void)
//
//
// LinkECP - Perform diagnostics
//
//
{
	nval32	fifo_size;

	outportb(Link_DCR, 0x20);
	outportb(Link_ECR, 0xd4);

	while ((inportb(Link_ECR) & 0x1) == 0)
		inportb(Link_FIFO);

	fifo_size = 0;

	while ((inportb(Link_ECR) & 0x2) == 0)
	{
		if (fifo_size >= 1024)
		{
			fifo_size = 0;

			break;
		}

		outportb(Link_FIFO, 0x00);

		fifo_size++;
	}

	if (fifo_size != 0)
	{
		while ((inportb(Link_ECR) & 0x1) == 0)
			inportb(Link_FIFO);
	}

	ECP_FIFO_Size = fifo_size;
}

void Link_Command_Diagnostics_ECP(void)
//
//
// LinkECP - Display diagnostics
//
//
{
	printf(
		"ECP diagnostics\n"
		"---------------\n"
		"- Port is configured at %03X\n",
		Config_LinkPort
	);

	if (ECP_FIFO_Size <= 0)
		printf("- Port is not present !\n");
	else
		printf(
			"- Port is present\n"
			"- FIFO size is %ld bytes\n"
			"- DSR register is %02X\n"
			"- DCR register is %02X\n",
			ECP_FIFO_Size,
			inportb(Link_DSR),
			inportb(Link_DCR)
		);
}

void Link_Receive_Initialise_ECP(void)
//
//
// LinkECP - Initialise receive
//
//
{
	outportb(Link_DCR, 0x20);
	outportb(Link_ECR, 0x74);
}

void Link_Receive_Block_ECP(nval8* data_buffer, nval32 number_of_bytes)
//
//
// LinkECP - Receive data block
//
//
{

#ifdef LINK_ASM

	asm volatile (
		"									\n\
											\n\
	movl	$0,%%eax		; // Permanently clear top 3 bytes of eax	\n\
	movl	%0,%%edi		; // Initialise byte pointer			\n\
	movl	%1,%%ecx		; // Initialise byte counter			\n\
	movl	%2,%%ebx		; // Initialise CRC value			\n\
											\n\
	movl	%6,%%edx		; // ECP FIFO size				\n\
											\n\
	cmpl	%%edx,%%ecx		; // At least one full FIFO to receive ?	\n\
	jl	JKE20			; // No, don't 'stream'				\n\
											\n\
	subl	%%edx,%%ecx		; // Pre-count FIFO block			\n\
											\n\
JKE10:											\n\
											\n\
	pushl	%%ecx			; // Save used registers			\n\
	pushl	%%edx			;						\n\
											\n\
	movl	%%edx,%%ecx		; // Initialise byte counter			\n\
											\n\
	movw	%4,%%dx			; // Address of ECR				\n\
											\n\
JKE11:											\n\
											\n\
	inb	%%dx,%%al		; // FIFO full ?				\n\
	testb	$0x02,%%al		;						\n\
	je	JKE11			; // Not yet, re-check				\n\
											\n\
JKE12:											\n\
											\n\
	movw	%5,%%dx			; // Address of FIFO				\n\
											\n\
	inb	%%dx,%%al		; // Copy byte from FIFO to buffer		\n\
	movb	%%al,(%%edi)		;						\n\
											\n\
	xorb	%%bl,%%al		; // Update CRC value				\n\
	shrl	$8,%%ebx		;						\n\
	xorl	%3(,%%eax,4),%%ebx	;						\n\
											\n\
	incl	%%edi			; // Update byte pointer			\n\
	decl	%%ecx			; // Update byte counter			\n\
											\n\
	jne	JKE12			; // Next byte					\n\
											\n\
	popl	%%edx			; // Restore used registers			\n\
	popl	%%ecx									\n\
											\n\
	subl	%%edx,%%ecx		; // Full FIFO left to receive ?		\n\
	jge	JKE10			; // Yes, next FIFO block			\n\
											\n\
	addl	%%edx,%%ecx		; // Correct FIFO block pre-count		\n\
											\n\
JKE20:											\n\
											\n\
	cmpl	$0,%%ecx		; // Any bytes left to receive ?		\n\
	jle	JKE90			; // No, ready					\n\
											\n\
	movw	%4,%%dx			; // Address of ECR				\n\
											\n\
JKE21:											\n\
											\n\
	inb	%%dx,%%al		; // FIFO contains at least 1 byte ?		\n\
	testb	$0x01,%%al		;						\n\
	jne	JKE21			; // Not yet, re-check				\n\
											\n\
	movw	%5,%%dx			; // Address of FIFO				\n\
											\n\
	inb	%%dx,%%al		; // Copy byte from FIFO to buffer		\n\
	movb	%%al,(%%edi)		;						\n\
											\n\
	xorb	%%bl,%%al		; // Update CRC value				\n\
	shrl	$8,%%ebx		;						\n\
	xorl	%3(,%%eax,4),%%ebx	;						\n\
											\n\
	incl	%%edi			; // Update byte pointer			\n\
	decl	%%ecx			; // Update byte counter			\n\
											\n\
	jne	JKE20			; // Next byte					\n\
											\n\
JKE90:											\n\
											\n\
	movl	%%ebx,%2		; // Register CRC value				\n\
											\n\
		"
		:
		: "m" (data_buffer),
		  "m" (number_of_bytes),
		  "m" (CRC32_Value),
		  "m" (CRC32_Table[0]),
		  "m" (Link_ECR),
		  "m" (Link_FIFO),
		  "m" (ECP_FIFO_Size)
		: "eax",
		  "ebx",
		  "ecx",
		  "edx",
		  "edi"
	   );

#else

	nval8*	data_walker;
	nval32	byte_counter;

	data_walker = data_buffer;
	byte_counter = number_of_bytes;

	while (byte_counter > 0)
	{
		CRC32_Value = CRC32_Table[(CRC32_Value ^ *data_walker) & 0xff] ^ (CRC32_Value >> 8);

		while (inportb(Link_ECR) & 0x01);

		*data_walker++ = inportb(Link_FIFO);

		byte_counter--;
	}

#endif

}

void Link_Diagnostics_BIDIR(void)
//
//
// LinkBIDIR - Perform diagnostics
//
//
{
	// Nothing at this time
}

void Link_Command_Diagnostics_BIDIR(void)
//
//
// LinkBIDIR - Display diagnostics
//
//
{
	printf(
		"BIDIR diagnostics\n"
		"-----------------\n"
		"- Port is configured at %03X\n"
		"- DSR register is %02X\n"
		"- DCR register is %02X\n",
		Config_LinkPort,
		inportb(Link_DSR),
		inportb(Link_DCR)
	);
}

void Link_Receive_Initialise_BIDIR(void)
//
//
// LinkBIDIR - Initialise receive
//
//
{
	outportb(Link_DCR, 0x20);
	outportb(Link_ECR, 0x34);
}

void Link_Receive_Block_BIDIR(nval8* data_buffer, nval32 number_of_bytes)
//
//
// LinkBIDIR - Receive data block
//
//
{

#ifdef LINK_ASM

	asm volatile (
		"									\n\
											\n\
	movl	$0,%%eax		; // Permanently clear top 3 bytes of eax	\n\
	movl	%0,%%edi		; // Initialise byte pointer			\n\
	movl	%1,%%ecx		; // Initialise byte counter			\n\
	movl	%2,%%ebx		; // Initialise CRC value			\n\
											\n\
	cmpl	$0,%%ecx		; // At least one byte to receive ?		\n\
	jle	JKB90			; // No						\n\
											\n\
	movw	%6,%%dx			; // Address of DCR				\n\
											\n\
	movb	$0x22,%%al		; // 'Not busy'					\n\
	outb	%%al,%%dx		;						\n\
											\n\
JKB10:											\n\
											\n\
	movw	%5,%%dx			; // Address of DSR				\n\
											\n\
JKB11:											\n\
											\n\
	inb	%%dx,%%al		; // Wait for 'strobe'				\n\
	testb	$0x40,%%al		;						\n\
	jne	JKB11			;						\n\
											\n\
	movw	%4,%%dx			; // Address of DATA				\n\
											\n\
	inb	%%dx,%%al		; // Copy byte from data port to buffer		\n\
	movb	%%al,(%%edi)		;						\n\
											\n\
	xorb	%%bl,%%al		; // Update CRC value				\n\
	shrl	$8,%%ebx		;						\n\
	xorl	%3(,%%eax,4),%%ebx	;						\n\
											\n\
	movw	%6,%%dx			; // Address of DCR				\n\
											\n\
	movb	$0x20,%%al		; // 'Busy'					\n\
	outb	%%al,%%dx		;						\n\
											\n\
	movw	%5,%%dx			; // Address of DSR				\n\
											\n\
JKB12:											\n\
											\n\
	inb	%%dx,%%al		; // Wait for 'not strobe'			\n\
	testb	$0x40,%%al		;						\n\
	je	JKB12			;						\n\
											\n\
	movw	%6,%%dx			; // Address of DCR				\n\
											\n\
	movb	$0x22,%%al		; // 'Not busy'					\n\
	outb	%%al,%%dx		;						\n\
											\n\
	incl	%%edi			; // Update byte pointer			\n\
	decl	%%ecx			; // Update byte counter			\n\
											\n\
	jne	JKB10			; // Next byte					\n\
											\n\
	movw	%5,%%dx			; // Address of DSR				\n\
											\n\
JKB90:											\n\
											\n\
	movl	%%ebx,%2		; // Register CRC value				\n\
											\n\
		"
		:
		: "m" (data_buffer),
		  "m" (number_of_bytes),
		  "m" (CRC32_Value),
		  "m" (CRC32_Table[0]),
		  "m" (Link_DATA),
		  "m" (Link_DSR),
		  "m" (Link_DCR)
		: "eax",
		  "ebx",
		  "ecx",
		  "edx",
		  "edi"
	   );

#else

#warning "Lack of non-assembly code for receive makes BIDIR mode unavailable"

#endif

}

void Link_Diagnostics(void)
//
//
// Link - Perform diagnostics
//
//
{
	switch (Config_Mode)
	{
	case MODE_ECP :
		Link_Diagnostics_ECP();
		break;
	case MODE_BIDIR :
		Link_Diagnostics_BIDIR();
		break;
	}
}

void Link_Command_Diagnostics(void)
//
//
// Link - Display diagnostics
//
//
{
	switch (Config_Mode)
	{
	case MODE_ECP :
		Link_Command_Diagnostics_ECP();
		break;
	case MODE_BIDIR :
		Link_Command_Diagnostics_BIDIR();
		break;
	}
}

void Link_Receive_Initialise(void)
//
//
// Link - Initialise receive
//
//
{
	switch (Config_Mode)
	{
	case MODE_ECP :
		Link_Receive_Initialise_ECP();
		break;
	case MODE_BIDIR :
		Link_Receive_Initialise_BIDIR();
		break;
	}
}

void Link_Receive_Block(nval8* data_buffer, nval32 number_of_bytes, bool do_synchronise)
//
//
// Link - Receive data block
//
//
{
	if (!do_synchronise)
	{
		while (true)
			inportb(Link_FIFO);
	}
	else
	{
		switch (Config_Mode)
		{
		case MODE_ECP :
			Link_Receive_Block_ECP(data_buffer, number_of_bytes);
			break;
		case MODE_BIDIR :
			Link_Receive_Block_BIDIR(data_buffer, number_of_bytes);
			break;
		}
	}
}

void Buffer_Initialise(nval8* data_buffer)
//
//
// Initialise data buffer access
//
//
{
	Buffer_Start = data_buffer;

	Buffer_Walker = Buffer_Start;
}

nval32 Buffer_Buffered(void)
//
//
// Returns number of bytes read from data buffer
//
//
{
	return (Buffer_Walker - Buffer_Start);
}

void Buffer_Byte1(nval8& input_value)
//
//
// Read 1-byte value from data buffer
//
//
{
	input_value = *Buffer_Walker++;
}

void Buffer_Byte2(nval16& input_value)
//
//
// Read 2-byte value from data buffer
//
//
{
	input_value = ((nval16) Buffer_Walker[0]) << 0 | ((nval16) Buffer_Walker[1]) << 8;

	Buffer_Walker += 2;
}

void Buffer_Byte4(nval32& input_value)
//
//
// Read 4-byte value from data buffer
//
//
{
	input_value = ((nval32) Buffer_Walker[0]) << 0 | ((nval32) Buffer_Walker[1]) << 8 | ((nval32) Buffer_Walker[2]) << 16 | ((nval32) Buffer_Walker[3]) << 24;

	Buffer_Walker += 4;
}

void Buffer_String(string& input_value)
//
//
// Read string value from data buffer
//
//
{
	nval8	value_length;

	Buffer_Byte1(value_length);

	input_value = string((char*) Buffer_Walker, value_length);

	Buffer_Walker += value_length;
}

void Buffer_Time(tm& input_value)
//
//
// Read 'link' time value from data buffer
//
//
{
	nval8	value;

	memset(&input_value, 0, sizeof(input_value));

	Buffer_Byte1(value);

	input_value.tm_year = value;

	Buffer_Byte1(value);

	input_value.tm_mon = value - 1;

	Buffer_Byte1(value);

	input_value.tm_mday = value;

	Buffer_Byte1(value);

	input_value.tm_hour = value;

	Buffer_Byte1(value);

	input_value.tm_min = value;

	Buffer_Byte1(value);

	input_value.tm_sec = value;

	input_value.tm_isdst = -1;
}

void Reset_Configuration(void)
//
//
// Reset configuration
//
//
{
	nval8	char_code;

	Config_Mode = MODE_BIDIR;

	Config_LinkPort = 0x378;

	Config_NoFileAccess = false;

	Config_NoSynchronise = false;

	Config_LeafLengthExt = 255;

	Config_LeafLengthName = 255;

	memset(Config_MapChar, 0, sizeof(Config_MapChar));

	for (char_code = 32; char_code <= 126; char_code++)
		Config_MapChar[char_code] = char_code;
}

void Read_Configuration_File(string file_path)
//
//
// Read configuration from file
//
//
{
	ifstream input_file(file_path.c_str());

	Reset_Configuration();

	if (!input_file)
		Issue_Fatal_OpenFile(file_path);

	int line_number = 0;

	while (input_file.good())
	{
		string	input_line;
		string	line_keyword;

		line_number++;

		getline(input_file, input_line);

		istringstream line_stream(input_line);

		line_stream >> line_keyword;

		if (line_keyword == "Mode")
		{
			string	config_mode;

			if (!(line_stream >> config_mode))
				Issue_Fatal_InLine(file_path, line_number, "missing or bad value");

			if (config_mode == "ECP")
				Config_Mode = MODE_ECP;
			else if (config_mode == "BIDIR")
				Config_Mode = MODE_BIDIR;
			else
				Issue_Fatal_InLine(file_path, line_number, "value must be ECP or BIDIR");
		}
		else if (line_keyword == "LinkPort")
		{
			line_stream.setf(ios_base::hex, ios_base::basefield);

			if (!(line_stream >> Config_LinkPort))
				Issue_Fatal_InLine(file_path, line_number, "missing or bad value");

			if (!(Config_LinkPort == 0x278 || Config_LinkPort == 0x378 || Config_LinkPort == 0x3BC))
				Issue_Fatal_InLine(file_path, line_number, "value must be (hex) 278, 378 or 3BC");
		}
		else if (line_keyword == "MapChar")
		{
			int	char_bad[2];
			int	char_good[2];

			while ((char_bad[0] = line_stream.get()) == ' ');

			char_bad[1] = line_stream.get();

			while ((char_good[0] = line_stream.get()) == ' ');

			char_good[1] = line_stream.get();

			if (char_bad[0] != '\'' || char_good[0] != '\'' || char_bad[1] == EOF || char_good[1] == EOF)
				Issue_Fatal_InLine(file_path, line_number, "value(s) must be : '<single char>");

			Config_MapChar[char_bad[1]] = char_good[1];
		}
		else if (line_keyword == "LeafLength")
		{
			if (!(line_stream >> Config_LeafLengthName >> Config_LeafLengthExt))
				Issue_Fatal_InLine(file_path, line_number, "missing or bad value(s)");

			if (Config_LeafLengthName <= 0)
				Config_LeafLengthName = 1;
			else if (Config_LeafLengthName >= 256)
				Config_LeafLengthName = 255;

			if (Config_LeafLengthExt <= 0)
				Config_LeafLengthExt = 1;
			else if (Config_LeafLengthExt >= 256)
				Config_LeafLengthExt = 255;
		}
		else if (line_keyword == "TypeToExt")
		{
			nval32	file_type;
			string	file_extension;

			line_stream.setf(ios_base::hex, ios_base::basefield);

			if (!(line_stream >> file_type >> file_extension))
				Issue_Fatal_InLine(file_path, line_number, "missing or bad value(s)");

			if (file_type < 0x1000)
				Input_Typemaps[file_type] = file_extension;
		}
		else if (line_keyword == "NoFileAccess")
			Config_NoFileAccess = true;
		else if (line_keyword == "NoSynchronise")
			Config_NoSynchronise = true;
		else if (!isalpha(line_keyword[0]))
			line_stream.ignore(input_line.size());
		else if (line_keyword != "")
			Issue_Fatal_InLine(file_path, line_number, "unknown keyword '" + line_keyword + "'");

		if (line_stream >> line_keyword)
			Issue_Fatal_InLine(file_path, line_number, "trailing characters");
	}

	input_file.close();
}

string Canonicalise_Path_Name(string path_name)
//
//
// Canonicalise a path name
//
//
{
	char	canonicalised_path[MAXIMUM_PATH_SIZE];

	CANONICALISE(path_name.c_str(), canonicalised_path);

	return canonicalised_path;
}

void Split_Path_Name(string path_name, string& directory_path, string& leaf_name)
//
//
// Split a path name into a directory path & leaf name
//
//
{
	nval32	pos;
	string	canonicalised_path;

	canonicalised_path = Canonicalise_Path_Name(path_name);

	if ((pos = canonicalised_path.find_last_of(SYS_SLASH)) == string::npos)
	{
		directory_path = "";

		leaf_name = canonicalised_path;
	}
	else
	{
		directory_path = canonicalised_path.substr(0, pos + 1);

		leaf_name = canonicalised_path.substr(pos + 1);
	}
}

string Map_Leaf_Name(string leaf_name)
//
//
// (Re-)map a leaf name (map characters as configured)
//
//
{
	string	normalised_leaf;

	for (string::const_iterator leaf_walker = leaf_name.begin(); leaf_walker != leaf_name.end(); leaf_walker++)
		normalised_leaf += Config_MapChar[(nval8) *leaf_walker];

	return normalised_leaf;
}

string Truncate_Leaf_Name(string leaf_name)
//
//
// Truncate a leaf name (discard characters as configured)
//
//
{
	nval32	pos;
	string	leaf_extension;

	if ((pos = leaf_name.find_last_of('.')) != string::npos)
	{
		leaf_extension = leaf_name.substr(pos);

		leaf_name = leaf_name.substr(0, pos);
	}

	if (leaf_name.size() > Config_LeafLengthName)
		leaf_name = leaf_name.substr(0, Config_LeafLengthName);

	if (leaf_extension.size() > (Config_LeafLengthExt + 1))
		leaf_extension = leaf_extension.substr(0, Config_LeafLengthExt + 1);

	return leaf_name + leaf_extension;
}

void Stamp_Local_File(string file_name, tm& file_time)
//
//
// Time-stamp a file
//
//
{
	utimbuf		time_stamp;
	time_t		target_time;

	target_time = mktime(&file_time);

	time_stamp.actime = target_time;
	time_stamp.modtime = target_time;

	utime(file_name.c_str(), &time_stamp);
}

void Receive_Object(nval32& object_type, nval8* data_buffer, nval32& number_of_bytes, nval32 maximum_number_of_bytes)
//
//
// Receive object
//
//
{
	bool	found_magic;
	nval8	header_buffer[6 * 4];
	nval8	magic_number[] =	{
						0x19, 0x15, 0x4A, 0x7A,
						0x69, 0xA9, 0xB3, 0x46,
						0x05, 0x2F, 0xC1, 0x29,
						0x85, 0x3F, 0x97, 0x55
			    		};

	Link_Receive_Block(header_buffer, 4 * 4, true);

	do
	{
		found_magic = (memcmp(header_buffer, magic_number, 4 * 4) == 0);

		if (!found_magic)
		{
			memmove(header_buffer, header_buffer + 1, 4 * 4 - 1);

			Link_Receive_Block(header_buffer + 4 * 4 - 1, 1, true);
		}
	}
	while (!found_magic);

	Link_Receive_Block(header_buffer + 4 * 4, 2 * 4, true);

	Buffer_Initialise(header_buffer + 4 * 4);

	Buffer_Byte4(object_type);

	Buffer_Byte4(number_of_bytes);

	if ((number_of_bytes >> 31) != 0)
		Issue_Fatal_Error("received object is too small");

	if (number_of_bytes > maximum_number_of_bytes)
		Issue_Fatal_Error("received object is too large");

	Link_Receive_Block(data_buffer, number_of_bytes, true);
}

bool Receive_File_Data(string file_path, nval32 file_size)
//
//
// Receive file data
//
// Returns true if CRC matched, else false
//
//
{
	bool	to_file;
	nval32	bytes_left;
	nval32	block_size;
	nval32	crc32_received;
	nval32	crc32_calculated;
	FILE*	output_file;

	to_file = !Config_NoFileAccess;

	if ((output_file = fopen(file_path.c_str(), "wb")) == NULL)
		Issue_Fatal_OpenFile(file_path);

	The_Stats.Initialise(file_size);

	bytes_left = file_size;

	CRC32_Value = 0xfffffffful;

	while (bytes_left > 0)
	{
		block_size = (bytes_left < LARGE_BUFFER_SIZE) ? bytes_left : LARGE_BUFFER_SIZE;

		Link_Receive_Block(Large_Data_Buffer, block_size, !Config_NoSynchronise);

		if (to_file)
			fwrite(Large_Data_Buffer, 1, block_size, output_file);

		bytes_left -= block_size;

		The_Stats.Update(block_size);
	}

	CRC32_Value = ~CRC32_Value;

	The_Stats.Finalise();

	crc32_calculated = CRC32_Value;

	Link_Receive_Block(Large_Data_Buffer, 4, true);

	Buffer_Initialise(Large_Data_Buffer);

	Buffer_Byte4(crc32_received);

	fclose(output_file);

	if (crc32_received == crc32_calculated)
	{
		if (Be_Verbose)
			cout << ", CRC passed" << endl;

		return true;
	}
	else
	{
		if (Be_Verbose)
			cout << ", CRC FAILED !!" << endl;

		remove(file_path.c_str());

		return false;
	}
}

bool Receive_AcornFile(nval8* file_info)
//
//
// Receive AcornFile object
//
// Returns true if CRC matched, else false
//
//
{
	nval32	file_load;
	nval32	file_exec;
	nval32	file_size;
	nval32	file_attr;
	nval32	file_type;
	tm	file_time;
	bool	stamp_file;
	string	file_leaf_name;
	string	target_leaf_name;
	string	target_path_name;

	Buffer_Initialise(file_info);

	Buffer_Byte4(file_size);

	Buffer_Byte4(file_load);

	Buffer_Byte4(file_exec);

	Buffer_Byte4(file_attr);

	Buffer_Time(file_time);

	Buffer_String(file_leaf_name);

	if ((file_load & 0xfff00000ul) == 0xfff00000ul)
	{
		stamp_file = true;

		file_type = (file_load >> 8) & 0xfff;
	}
	else
	{
		stamp_file = false;

		file_type = 0x1000;
	}

	target_leaf_name = Map_Leaf_Name(file_leaf_name);

	if (file_type < 0x1000)
	{
		map<nval32,string>::const_iterator mapping = Input_Typemaps.find(file_type);

		if (mapping != Input_Typemaps.end())
			target_leaf_name += "." + mapping->second;
	}

	target_leaf_name = Truncate_Leaf_Name(target_leaf_name);

	target_path_name = Canonicalise_Path_Name(target_leaf_name);

	if (Be_Verbose)
		cout << "Receiving Acorn file '" << file_leaf_name << "' as '" << target_path_name << "', " << file_size << " bytes" << flush;

	if (!Receive_File_Data(target_path_name, file_size))
		return false;

	if (stamp_file)
		Stamp_Local_File(target_path_name, file_time);

	return true;
}

bool Receive_WinnixFile(int object_type, nval8* file_info)
//
//
// Receive WindowsFile or UnixFile object
//
// Returns true if CRC matched, else false
//
//
{
	nval32	file_size;
	nval16	file_attr;
	tm	file_time;
	string	file_leaf_name;
	string	target_leaf_name;
	string	target_path_name;

	Buffer_Initialise(file_info);

	Buffer_Byte4(file_size);

	Buffer_Byte2(file_attr);

	Buffer_Time(file_time);

	Buffer_String(file_leaf_name);

	target_leaf_name = Map_Leaf_Name(file_leaf_name);

	target_leaf_name = Truncate_Leaf_Name(target_leaf_name);

	target_path_name = Canonicalise_Path_Name(target_leaf_name);

	if (Be_Verbose)
		cout << "Receiving " << ((object_type == OBJECT_WINDOWSFILE) ? "Windows" : "Unix") << " file '" << file_leaf_name << "' as '" << target_path_name << "', " << file_size << " bytes" << flush;

	if (!Receive_File_Data(target_path_name, file_size))
		return false;

	Stamp_Local_File(target_path_name, file_time);

	return true;
}

void Receive_Info(nval8* info_info)
//
//
// Receive Info object
//
//
{
	nval32	info_size;
	nval32	info_type;

	Buffer_Initialise(info_info);

	Buffer_Byte4(info_size);

	if (info_size > LARGE_BUFFER_SIZE)
		info_size = LARGE_BUFFER_SIZE;

	Link_Receive_Block(Large_Data_Buffer, info_size, !Config_NoSynchronise);

	Buffer_Initialise(Large_Data_Buffer);

	Buffer_Byte4(info_type);

	switch (info_type)
	{
	case 1 :
		{
			cout << ((char*) &Large_Data_Buffer[4]) << endl;

			fflush(stdout);
		}
		break;
	}
}

int main_actual(int argument_count, char** argument_list)
{
	nval32	file_count = 0;
	nval32	object_type;
	nval32	object_bytes;
	bool	received_null;
	bool	do_diagnostics = false;
	nval32	bad_file_count = 0;
	int	argument_index;
	char*	argument_value;

	//
	// Announce
	//

	cout << "\n" CODE_NAME " " CODE_VERSION ", " CODE_COPYRIGHT "\n" << endl;

	//
	// Process arguments
	//

	argument_index = 0;

	while (++argument_index < argument_count)
	{
		argument_value = argument_list[argument_index];

		if (argument_value[0] == '-')
		{
			switch (argument_value[1])
			{
			case 'd' :
				{
					if (argument_value[2] != '\0')
						Issue_Usage_Error(MAKE_STRING("syntax: -d, used: " << argument_value));

					do_diagnostics = true;
				}
				break;
			case 'q' :
				{
					if (argument_value[2] != '\0')
						Issue_Usage_Error(MAKE_STRING("syntax: -q, used: " << argument_value));

					Be_Verbose = false;
				}
				break;
			case 'v' :
				{
					if (argument_value[2] != '\0')
						Issue_Usage_Error(MAKE_STRING("syntax: -v, used: " << argument_value));

					Be_Verbose = true;
				}
				break;
			default:
				{
					Issue_Usage_Error(MAKE_STRING("unknown option: " << argument_value));
				}
				break;
			}
		}
		else
			Issue_Usage_Error(MAKE_STRING("superfluous argument: " << argument_value));
	}

	//
	// Run
	//

	string leaf_name;
	string directory_path;

	Split_Path_Name(argument_list[0], directory_path, leaf_name);

	Read_Configuration_File(MAKE_STRING(directory_path << "link.cnf"));

	Link_DATA = Config_LinkPort + PPORT_OFF_DATA;
	Link_DSR  = Config_LinkPort + PPORT_OFF_DSR ;
	Link_DCR  = Config_LinkPort + PPORT_OFF_DCR ;
	Link_FIFO = Config_LinkPort + PPORT_OFF_FIFO;
	Link_ECR  = Config_LinkPort + PPORT_OFF_ECR ;

#ifdef UNIX

	if (iopl(3) != 0)
		Issue_Fatal_Error("cannot get access to I/O ports");

#endif
	Link_Diagnostics();

	if (do_diagnostics)
	{
		Link_Command_Diagnostics();

		return 0;
	}

	Small_Data_Buffer = new nval8[SMALL_BUFFER_SIZE];
	Large_Data_Buffer = new nval8[LARGE_BUFFER_SIZE];

	Link_Receive_Initialise();

	if (Be_Verbose)
		cout << "Receiving ...\n" << endl;

	received_null = false;

	do
	{
		Receive_Object(object_type, Small_Data_Buffer, object_bytes, SMALL_BUFFER_SIZE);

		switch (object_type)
		{
		case OBJECT_NULL :
			{
				received_null = true;
			}
			break;
		case OBJECT_ACORNFILE :
			{
				if (!Receive_AcornFile(Small_Data_Buffer))
					bad_file_count++;

				file_count++;
			}
			break;
		case OBJECT_WINDOWSFILE :
		case OBJECT_UNIXFILE :
			{
				if (!Receive_WinnixFile(object_type, Small_Data_Buffer))
					bad_file_count++;

				file_count++;
			}
			break;
		case OBJECT_INFO :
			{
				Receive_Info(Small_Data_Buffer);
			}
			break;
		default :
			{
				Issue_Fatal_Error("received object has invalid type");
			}
		}
	}
	while (!received_null);

	if (Be_Verbose)
		cout << "\n" << file_count << " file(s) received, " << bad_file_count << " bad" << endl;

	return 0;
}

void my_new_handler(void)
{
	throw bad_alloc();
}

int main(int argument_count, char** argument_list)
{
	int	result = 0;

	set_new_handler(my_new_handler);

	try
	{
		result = main_actual(argument_count, argument_list);
	}
	catch (bad_alloc)
	{
		Issue_Fatal_Error("unexpectedly out of memory");
	}

	return result;
}



