
/*

GoXXCio.cpp

IO utils for GoMMC and GoSDC

Version : 0.82
Date    : 3 January 2026
Author  : John Kortink, (c) Zeridajh 2004..2026

*/

#include <list>
#include <array>
#include <ctime>
#include <cstdio>
#include <string>
#include <vector>
#include <cstdlib>
#include <cstring>
#include <fstream>
#include <sstream>
#include <iostream>
#include <algorithm>
#include <filesystem>
#include <functional>
#include "lib_types.h"
#include "lib_data_block.h"
#include "lib_make_string.h"
#include "lib_acorn_drive.h"
#include "lib_neat_message.h"

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Non-code
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef GOMMC

#define HW_NAME		"GoMMC"
#define CODE_NAME	"GoMMCio"

#endif

#ifdef GOSDC

#define HW_NAME		"GoSDC"
#define CODE_NAME	"GoSDCio"

#endif

#define CODE_VERSION	"0.82 (3 January 2026)"
#define CODE_COPYRIGHT	"Copyright (c) Zeridajh 2004..2026"

#undef VERBOSE

#ifdef UNIX

#define SYS_SLASH	"/"
#define SYS_DEVFMT	"%s"
#define SYS_DOPTSYN	"<device>"
#define SYS_DOPTUSE	"<device>  : flash card is device <device>"
#define SYS_FINISH	{ std::cout << std::endl; }

#define OFFT_64		off_t
#define FOPEN_64	fopen
#define FSEEK_64	fseeko
#define STRCASECMP	strcasecmp

#endif

#ifdef WINDOWS_CW

#define SYS_SLASH	"\\"
#define SYS_DEVFMT	"%s"
#define SYS_DOPTSYN	"<drive>"
#define SYS_DOPTUSE	"<drive>   : flash card is drive <drive>"
#define SYS_FINISH	{}

#define OFFT_64		off_t
#define FOPEN_64	fopen
#define FSEEK_64	fseeko
#define STRCASECMP	strcasecmp

#endif

#ifdef WINDOWS_VS

#define SYS_SLASH	"\\"
#define SYS_DEVFMT	"\\\\.\\%s:"
#define SYS_DOPTSYN	"<drive>"
#define SYS_DOPTUSE	"<drive>   : flash card is drive <drive>"
#define SYS_FINISH	{}

#define OFFT_64		__int64
#define FOPEN_64	fopen
#define FSEEK_64	_fseeki64
#define STRCASECMP	_stricmp

#include <direct.h>

#endif

#define CRD_ADDR_GHD	0x00000000
#define CRD_ADDR_CAT	0x00000200

#define CRD_SIGN_GHD	"GoMMCCat"
#define CRD_SIGN_OHD	"GoMMCEnt"

#define GLOBAL_HEADER_SIZE	512
#define CATALOGUE_ENTRY_SIZE	64
#define OBJECT_HEADER_SIZE	512

#define FILING_SYSTEMS		12

#define CARD_SECTOR_SIZE	512

#define FILLER_BYTE_CARD	nval8(0x7a)
#define FILLER_BYTE_DISC	nval8(0xe5)
#define FILLER_BYTE_FILE	nval8(0x00)

#define CARD_NIBBLE_SIZE	CARD_SECTOR_SIZE

#define CHUNK_BUFFER_SIZE	(1 * 1024 * 1024)

#define CACHE_BUFFER_SIZE	(4 * 1024 * 1024)

#define	ROUND_DOWN(v,m)		(((v) / (m)) * (m))

#define	ROUND_UP(v,m)		((((v) + (m) - 1) / (m)) * (m))

#define SIZE_ON_CARD(v)		ROUND_UP(v, CARD_SECTOR_SIZE)

#define VALID_FS_NUMBER(n)	((n >= 1 && n <= FILING_SYSTEMS) || (n >= 81 && n <= 80 + FILING_SYSTEMS))

#define FS_NUMBER_TO_NAME(n)	(!VALID_FS_NUMBER(n) ? std::string("<unknown FS>") : Filing_System_Info.at((n < 80) ? n - 1 : n - 81).name + ((n < 80) ? " (" HW_NAME ")" : ""))

#define UE_NUMBER_TO_NAME(n)	("UE #" + std::to_string(n))

enum process_type_t
{
	process_type_generic,
	process_type_dfs_disc,
	process_type_adfs_disc,
	process_type_tape_raw,
	process_type_tape_cooked,
	process_type_file_cooked
};

enum object_type_t
{
	object_type_fs,
	object_type_tool,
	object_type_medium,
	object_type_dfs_disc,
	object_type_adfs_disc,
	object_type_hadfs_disc,
	object_type_tape,
	object_type_ue,
	object_type_file,
	object_type_unknown
};

enum object_part_t
{
	objecr_part_file,
	objecr_part_data,
	objecr_part_made
};

struct object_type_info
{
	const nval16		int_id;
	const object_type_t	ext_id;
	const char*		name;
};

struct filing_system_info
{
	const int		number;
	const std::string	name;
	const nval32		crc_o;
	const nval32		crc_p;
	const nval16		skip;
	const nval16		note;
};

struct copy_map_entry
{
	const nval32	size;
	const zval32	step;
	const int	jump;
};

struct Option_List
{
	bool	shrink;
	bool	blowup;
	bool	compact;
	bool	deinterlace;
	bool	reinterlace;
	nval32	blowup_size;

	Option_List() : shrink(false), blowup(false), compact(false), deinterlace(false), reinterlace(false) {};
};

struct Object_Part
{
	const object_part_t	type;
	const nval32		size;
	const nval32		skip;
	const std::string	name;
	nval8* const		data;
	const nval8		byte;

	Object_Part(const std::string& a_name, nval32 a_size, nval32 a_skip = 0) : type(objecr_part_file), size(a_size), skip(a_skip), name(a_name), data(NULL), byte(0) {};
	Object_Part(nval8* a_data, nval32 a_size) : type(objecr_part_data), size(a_size), skip(0), name(""), data(a_data), byte(0) {};
	Object_Part(nval8 a_byte, nval32 a_size) : type(objecr_part_made), size(a_size), skip(0), name(""), data(NULL), byte(a_byte) {};
};

struct Object_Plan
{
	std::list<Object_Part>	parts;

	void Add_Part(const Object_Part& part)
	{
		parts.push_back(part);
	}
};

class Global_Header
{

//
// Variables
//

public :

	std::string	signature;
	nval32		cat_free_ptr;
	nval32		dat_free_ptr;

//
// Functions
//

public :

	void Import(const nval8* raw)
	{
		zeridajh::CData_Block data_block(raw);

		signature = data_block.String(0)(8);
		cat_free_ptr = data_block.NVal32(16);
		dat_free_ptr = data_block.NVal32(20);
	}

	void Export(nval8* raw) const
	{
		zeridajh::CData_Block data_block(raw);

		data_block.Bytes(GLOBAL_HEADER_SIZE, 0) = (nval8) 0x00;

		data_block.String(0) = signature;
		data_block.NVal32(16) = cat_free_ptr;
		data_block.NVal32(20) = dat_free_ptr;
	}

};

class Catalogue_Entry
{

//
// Types
//

private :

	enum : nval8
	{
		property_deleted = 0x01
	};

protected :

	enum : nval8
	{
		type_none   = 0,
		type_fs     = 1,
		type_tool   = 2,
		type_medium = 3,
		type_ue     = 4,
		type_file   = 5
	};

	enum : nval8
	{
		subtype_none         = 0,
		subtype_medium_dfs   = 1,
		subtype_medium_adfs  = 2,
		subtype_medium_cfs   = 3,
		subtype_medium_hadfs = 4
	};

public :

	enum : int
	{
		name_size_full = 48,
		name_size_file = 37
	};

//
// Variables
//

private :

	nval8	type;
	nval8	subtype;
	nval8	property;
	nval32	card_address;

protected :

	std::string	name;
	nval32		size_in_bytes;

//
// Functions
//

private :

	bool Compare_With(const Catalogue_Entry& other, bool clash_else_match) const;

protected :

	Catalogue_Entry() = default;

	Catalogue_Entry(nval8 a_type, nval8 a_subtype, const std::string& a_name, nval32 a_size_in_bytes)
	: type(a_type), subtype(a_subtype), property(0), card_address(0), name(a_name), size_in_bytes(a_size_in_bytes)
	{};

public :

	virtual ~Catalogue_Entry() {};

	static Catalogue_Entry*		Create(const nval8* raw);
	virtual void			Import(const nval8* raw);
	virtual void			Export(nval8* raw) const;

	nval16			Type_Id() const;
	bool			Deleted() const;
	nval32			Card_Address() const;
	nval32			Allocated_Bytes() const;
	virtual nval32		Populated_Bytes() const;
	void			Card_Address(nval32 a_card_address);
	bool			operator<(const Catalogue_Entry& other) const;
	bool			Matches_With(const Catalogue_Entry& other) const;
	bool			Clashes_With(const Catalogue_Entry& other) const;
	void			Delete();
	void			Rename(const std::string& a_name);
	virtual std::string	Stream(const std::string& format) const;

};

class Catalogue_Entry_FS : public Catalogue_Entry
{

public :

	Catalogue_Entry_FS() = default;

	Catalogue_Entry_FS(nval8 a_subtype, const std::string& a_name, nval32 a_size_in_bytes = 0)
	: Catalogue_Entry(type_fs, a_subtype, a_name, a_size_in_bytes)
	{};

};

class Catalogue_Entry_Tool : public Catalogue_Entry
{

public :

	Catalogue_Entry_Tool() = default;

	Catalogue_Entry_Tool(const std::string& a_name, nval32 a_size_in_bytes = 0)
	: Catalogue_Entry(type_tool, subtype_none, a_name, a_size_in_bytes)
	{};

};

class Catalogue_Entry_Medium : public Catalogue_Entry
{

public :

	Catalogue_Entry_Medium(const std::string& a_name, nval32 a_size_in_bytes = 0)
	: Catalogue_Entry(type_medium, subtype_none, a_name, a_size_in_bytes)
	{};

};

class Catalogue_Entry_DFS_Disc : public Catalogue_Entry
{

public :

	Catalogue_Entry_DFS_Disc() = default;

	Catalogue_Entry_DFS_Disc(const std::string& a_name, nval32 a_size_in_bytes = 0)
	: Catalogue_Entry(type_medium, subtype_medium_dfs, a_name, a_size_in_bytes)
	{};

};

class Catalogue_Entry_ADFS_Disc : public Catalogue_Entry
{

public :

	Catalogue_Entry_ADFS_Disc() = default;

	Catalogue_Entry_ADFS_Disc(const std::string& a_name, nval32 a_size_in_bytes = 0)
	: Catalogue_Entry(type_medium, subtype_medium_adfs, a_name, a_size_in_bytes)
	{};

};

class Catalogue_Entry_Tape : public Catalogue_Entry
{

public :

	Catalogue_Entry_Tape() = default;

	Catalogue_Entry_Tape(const std::string& a_name, nval32 a_size_in_bytes = 0)
	: Catalogue_Entry(type_medium, subtype_medium_cfs, a_name, a_size_in_bytes)
	{};

};

class Catalogue_Entry_HADFS_Disc : public Catalogue_Entry
{

public :

	Catalogue_Entry_HADFS_Disc() = default;

	Catalogue_Entry_HADFS_Disc(const std::string& a_name, nval32 a_size_in_bytes = 0)
	: Catalogue_Entry(type_medium, subtype_medium_hadfs, a_name, a_size_in_bytes)
	{};

};

class Catalogue_Entry_UE : public Catalogue_Entry
{

public :

	Catalogue_Entry_UE() = default;

	Catalogue_Entry_UE(nval8 a_subtype, const std::string& a_name, nval32 a_size_in_bytes = 0)
	: Catalogue_Entry(type_ue, a_subtype, a_name, a_size_in_bytes)
	{};

};

class Catalogue_Entry_File : public Catalogue_Entry
{

private :

	nval16	load_address;
	nval16	exec_address;
	nval24	current_size;

public :

	Catalogue_Entry_File() = default;

	Catalogue_Entry_File(const std::string& a_name, nval32 a_size_in_bytes = 0, nval16 a_load_address = 0, nval16 an_exec_address = 0, nval24 a_current_size = 0)
	: Catalogue_Entry(type_file, subtype_none, a_name, a_size_in_bytes), load_address(a_load_address), exec_address(an_exec_address), current_size(a_current_size)
	{};

	void	Import(const nval8* raw);
	void	Export(nval8* raw) const;

	nval32		Populated_Bytes() const;
	std::string	Stream(const std::string& format) const;

};

class Catalogue_Entry_Unknown : public Catalogue_Entry
{

public :

	Catalogue_Entry_Unknown() = default;

};

class Object_Header
{

//
// Variables
//

public :

	std::string		signature;
	const Catalogue_Entry*	catalogue_entry;

//
// Functions
//

public :

	void Export(nval8* raw) const
	{
		zeridajh::CData_Block data_block(raw);

		data_block.Bytes(OBJECT_HEADER_SIZE, 0) = (nval8) 0x00;

		data_block.String() = signature;

		catalogue_entry->Export(&raw[16]);
	}

};

class Script_Line
{

//
// Variables
//

private :

	const int		source;
	const int		number;
	const std::string	original;
	std::string		consumed;

//
// Functions
//

private :

	void Skip()
	{
		const auto cursor = consumed.find_first_not_of(" \t");

		if (cursor == std::string::npos)
			consumed.erase();
		else
			consumed.erase(0, cursor);
	}

public :

	Script_Line(int a_source, int a_number, const std::string& text) : source(a_source), number(a_number), original(text), consumed(text) {};

	bool Ignore() const
	{
		return original.empty() || original.front() == '#';
	}

	std::string Location() const
	{
		return std::to_string(source) + "." + std::to_string(number);
	}

	bool Empty()
	{
		Skip();

		return consumed.empty();
	}

	bool Pass(const std::string& literal)
	{
		Skip();

		if (consumed.substr(0, literal.length()) == literal)
		{
			consumed.erase(0, literal.length());

			return true;
		}

		return false;
	}

	int Pass(const std::vector<const char*>& literals)
	{
		for (size_t index = 0; index < literals.size(); index++)
			if (Pass(literals.at(index)))
				return index + 1;

		return 0;
	}

	std::string Pass()
	{
		Skip();

		const auto cursor = consumed.find_first_of(" \t");

		const std::string nonliteral = (cursor == std::string::npos) ? consumed : consumed.substr(0, cursor);

		consumed.erase(0, nonliteral.length());

		return nonliteral;
	}

	std::string Left()
	{
		Skip();

		const std::string left = consumed;

		consumed.erase();

		return left;
	}

};

class Command
{

//
// Types
//

protected :

	typedef std::string object_name_t;

	typedef nval8 object_number_t;

	typedef nval32 object_address_t;

	struct file_spec_t
	{
		std::string	name;
		nval32		size;
	};

	typedef nval32 disc_size_t;

	typedef nval32 file_room_t;

//
// Variables
//

protected :

	Option_List		options;
	Catalogue_Entry*	old_entry;
	Catalogue_Entry*	new_entry;
	Script_Line&		script_line;

//
// Functions
//

protected :

	Command(Script_Line& a_script_line);

	virtual ~Command();

	std::string		Arg(const char* what, bool left_else_pass);
	object_type_t		Arg_Object_Type();
	object_name_t		Arg_Object_Name(size_t limit);
	object_number_t		Arg_Object_Number(const char* what);
	object_address_t	Arg_Object_Address(const char* what);
	file_spec_t		Arg_File_Name(bool check);
	std::string		Arg_Directory_Name();
	disc_size_t		Arg_Disc_Size(const char* what);
	file_room_t		Arg_File_Room();
	int			Arg_Choice(const char* what, const std::vector<const char*>& choices);
	void			Arg_Finish();

public :

	virtual void	Update() {};
	virtual void	Execute() {};
	virtual nval32	Size_On_Card() const { return 0; };

};

class Command_Do : public Command
{

public :

	Command_Do(Script_Line& script_line);

};

class Command_Add : public Command
{

private :

	const Catalogue_Entry*	cat_entry;
	std::string		source_file;
	nval32			source_skip;
	process_type_t		process_type;

private :

	void	Process_Disc(const std::string& file_name, bool adfs, Option_List& options, nval32& old_disc_size, nval32& new_disc_size, nval8*& new_disc_data, bool test) const;
	void	Process_Disc(const std::string& file_name, bool adfs, Option_List& options, nval32& new_disc_size) const;
	void	Process_Disc(const std::string& file_name, bool adfs, Option_List& options, nval32& old_disc_size, nval32& new_disc_size, nval8*& new_disc_data) const;
	void	Process_Tape(const std::string& file_name, nval32& new_tape_size, nval8*& new_tape_data, bool test) const;
	void	Process_Tape(const std::string& file_name, nval32& new_tape_size) const;
	void	Process_Tape(const std::string& file_name, nval32& new_tape_size, nval8*& new_tape_data) const;

public :

	Command_Add(Script_Line& script_line);

	void	Update();
	void	Execute();
	nval32	Size_On_Card() const;

};

class Command_New : public Command
{

private :

	const Catalogue_Entry*	cat_entry;
	nval32			target_size;
	process_type_t		process_type;

public :

	Command_New(Script_Line& script_line);

	void	Update();
	void	Execute();
	nval32	Size_On_Card() const;

};

class Command_Delete : public Command
{

private :

	const Catalogue_Entry* cat_entry;

public :

	Command_Delete(Script_Line& script_line);

	void	Update();
	void	Execute();

};

class Command_Rename : public Command
{

private :

	const Catalogue_Entry*	cat_entry;
	std::string		target_name;

public :

	Command_Rename(Script_Line& script_line);

	void	Update();
	void	Execute();

};

class Command_Backup : public Command
{

private :

	std::string directory_name;

public :

	Command_Backup(Script_Line& script_line);

	void	Update();
	void	Execute();

};

class Command_Extract : public Command
{

private :

	const Catalogue_Entry*	cat_entry;
	std::string		target_file;
	process_type_t		process_type;

public :

	Command_Extract(Script_Line& script_line);

	void	Update();
	void	Execute();

};

class Command_List
{

//
// Variables
//

private :

	std::list<Command*>	command_list;
	bool			nothing_to_do;
	nval32			nr_deletes;
	nval32			nr_renames;
	nval32			nr_derives;
	nval32			new_objects_count;
	nval32			new_objects_bytes;

//
// Functions
//

public :

	Command_List();

	void	Append(Script_Line& script_line);
	void	Evaluate();
	void	Update();
	void	Execute();
	nval32	Delta_Catalogue() const;
	nval32	Delta_Objects() const;

};

class Card_Storage
{

//
// Variables
//

private :

	std::string	card_device;
	nval32		cache_size;
	nval8*		cache_buffer;
	nval32		cache_fill_lo;
	nval32		cache_fill_hi;
	nval32		cache_card_address;

//
// Functions
//

private :

	nval32	Read(FILE* io_file, nval32& card_address, nval8*& data_buffer, nval32 bytes_to_transfer);
	void	Read(nval32 card_address, nval8* data_buffer, nval32 bytes_to_transfer);
	nval32	Write(FILE* io_file, nval32& card_address, const nval8*& data_buffer, nval32 bytes_to_transfer);
	void	Write(nval32 card_address, const nval8* data_buffer, nval32 bytes_to_transfer);
	nval8*	Cache(nval32 card_address, nval32 bytes_to_cache);

public :

	Card_Storage(const std::string& card_spec, nval32 a_cache_size);

	void	Flush();
	nval8*	Write_Cached(nval32 card_address, const nval8* buffer, nval32 bytes_to_write);
	void	Read_Object(nval32 card_address, nval32 size_in_bytes, const Object_Plan& object_plan);
	void	Write_Object(nval32 card_address, nval32 size_in_bytes, const Object_Plan& object_plan);
	void	Import_Data(nval32 card_address, nval8* buffer, nval32 bytes_to_read);
	void	Export_Data(nval32 card_address, const nval8* buffer, nval32 bytes_to_write);

};

class Card_Global_Header
{

//
// Variables
//

private :

	bool		dirty;
	Global_Header	contents;
	const nval32	card_address;
	const nval32	size_in_bytes;

//
// Functions
//

public :

	Card_Global_Header(nval32 a_card_address, nval32 a_size_in_bytes, bool make_up_header);

	void	Flush();
	nval32	Catalogue_Entries() const;
	nval32	Free_Data_Pointer() const;
	void	Catalogue_Entries(nval32 new_value);
	void	Free_Data_Pointer(nval32 new_value);

};

class Card_Catalogue
{

//
// Variables
//

private :

	bool				dirty;
	std::vector<Catalogue_Entry*>	contents;
	const nval32			card_address;
	nval32				objects_base;
	nval32				objects_size;
	nval32				free_pointer;
	nval32				current_entries;

//
// Functions
//

public :

	Card_Catalogue(nval32 a_card_address, nval32 size_in_entries);

	~Card_Catalogue();

	void			Flush();
	nval32			Base_Address() const;
	nval32			End_Address() const;
	nval32			Size_In_Bytes() const;
	nval32			Size_In_Entries() const;
	nval32			Objects_Base_Address() const;
	nval32			Objects_End_Address() const;
	nval32			Objects_Size_In_Bytes() const;
	void			Scan(std::function<void(const Catalogue_Entry*)> processor) const;
	void			List() const;
	const Catalogue_Entry*	Check_Entry_Present(const Catalogue_Entry* old_entry, const Script_Line& script_line) const;
	void			Check_Entry_Absent(const Catalogue_Entry* new_entry, const Script_Line& script_line) const;
	void			Add_Entry(Catalogue_Entry* new_entry);
	void			Delete_Entry(const Catalogue_Entry* cat_entry);
	void			Rename_Entry(const Catalogue_Entry* cat_entry, const std::string& new_name);

};

class Card_Objects
{

//
// Variables
//

private :

	bool		dirty;
	nval32		card_address;
	nval32		existing_bytes;
	const nval32	additional_bytes;

//
// Functions
//

public :

	Card_Objects(nval32 bytes_to_add);

	void	Flush();
	nval32	Base_Address() const;
	nval32	Size_In_Bytes() const;
	void	Object_To_Card(const Catalogue_Entry& cat_entry, const Object_Plan& object_plan);

};

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Globals
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

OFFT_64			The_Area;
Option_List		The_Options;
Command_List		The_Commands;
Card_Storage*		The_Storage = NULL;
Card_Objects*		The_Old_Objects = NULL;
Card_Objects*		The_New_Objects = NULL;
Card_Global_Header*	The_Global_Header = NULL;
Card_Catalogue*		The_Old_Catalogue = NULL;
Card_Catalogue*		The_New_Catalogue = NULL;

zeridajh::Make_String	make_string;

const std::array<const object_type_info, 9> Object_Type_Info =
{{
	{ 0x0100, object_type_fs        , "FS"         },
	{ 0x0200, object_type_tool      , "TOOL"       },
	{ 0x0300, object_type_medium    , "MEDIUM"     },
	{ 0x0301, object_type_dfs_disc  , "DFS DISC"   },
	{ 0x0302, object_type_adfs_disc , "ADFS DISC"  },
	{ 0x0303, object_type_tape      , "TAPE"       },
	{ 0x0304, object_type_hadfs_disc, "HADFS DISC" },
	{ 0x0400, object_type_ue        , "UE"         },
	{ 0x0500, object_type_file      , "FILE"       }
}};

const std::array<const filing_system_info, 12> Filing_System_Info =
{{
	{  1, "DNFS 1.20",    0x7ccc9f1aL, 0xaccc86b2L, 0x0, 0x0 },
	{  2, "DFS 2.26",     0x2dcaf65bL, 0xc82a53dfL, 0x0, 0x0 },
	{  3, "ADFS 1.30",    0xd16a5abfL, 0x946f788cL, 0x0, 0x0 },
	{  4, "DFS 2.24",     0xec932606L, 0x956c2d33L, 0x0, 0x0 },
	{  5, "ADFS 1.50",    0xea032ec9L, 0x3367e5f7L, 0x0, 0x0 },
	{  6, "DFS 2.45",     0x9a64c2a3L, 0x137cf3bbL, 0x0, 0x0 },
	{  7, "ADFS 2.03",    0x6bb8c053L, 0x7abdca61L, 0x0, 0x0 },
	{  8, "WE DFS 1.44",  0xd31610c8L, 0x2816553eL, 0x0, 0x0 },
	{  9, "ACP DFS 1.00", 0x2502a5bcL, 0x5390cd46L, 0x0, 0x0 },
	{ 10, "ACP DFS 2.20", 0x454f5602L, 0x459ddfd3L, 0x0, 0x0 },
	{ 11, "ADFS 1.00",    0xb3c09d1eL, 0xbe045d4dL, 0x0, 0x0 },
	{ 12, "HADFS X.YZ",   0x00000000L, 0xbae0a2afL, 0x9, 0x6 }
}};

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Messages
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void Issue_Message(const std::string& severity, const std::string& message)
{
	std::cerr << "\n" << zeridajh::Neat_Message(CODE_NAME ": ", severity + ": " + message);
}

void Issue_Warning(const std::string& message)
{
	Issue_Message("warning", message);
}

void Issue_Fatal_Error(const std::string& message)
{
	Issue_Message("fatal error", message);

	SYS_FINISH;

	exit(1);
}

std::string Line_Message(const Script_Line& line, const std::string& message)
{
	std::ostringstream stream;

	stream << "line " << line.Location() << ": " << message;

	return stream.str();
}

void Issue_Warning(const Script_Line& line, const std::string& message)
{
	Issue_Warning(Line_Message(line, message));
}

void Issue_Fatal_Error(const Script_Line& line, const std::string& message)
{
	Issue_Fatal_Error(Line_Message(line, message));
}

void Issue_Usage_Error(const std::string& message)
{
	std::cerr << "\n" << zeridajh::Neat_Message(CODE_NAME ": ", "usage error: " + message);

	std::cerr <<

	"\n"
	"Usage: " CODE_NAME " [option...] [script file...]\n"
	"\n"
	"[option] is one of :\n"
	"-d" SYS_DOPTUSE "\n"
#ifdef GOSDC
	"-a<number>  : select area <number> (1-8)\n"
#endif
	"-f          : discard old contents of flash card\n"
#ifdef GOMMC
	"-x          : fake a 1 MB (MMCFO) format\n"
#endif
	"-l          : list the new flash card catalogue\n"
	"-c \"<line>\" : add single script line <line>\n"
	"\n"
	"[x] means x is optional\n"
	"<x> means x is required\n"
	"x... means one or more times x\n"

	;

	SYS_FINISH;

	exit(1);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Miscellaneous
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

std::string In_Hex(nval32 value, int digits)
//
//
// Return value as hexadecimal string
//
//
{
	std::ostringstream stream;

	stream << std::hex << std::setw(digits) << std::setfill('0') << value;

	return stream.str();
}

nval32 Crc_32(const nval8* data, nval32 bytes)
//
//
// Return CRC-32-IEEE of a data block
//
//
{
	nval32 crc = 0;

	while (bytes-- > 0)
	{
		crc ^= ((nval32) *data++ << 24);

		for (int bit = 0; bit < 8; bit++)
		{
			if (crc & 0x80000000)
				crc = (crc << 1) ^ 0x04C11DB7;
			else
				crc <<= 1;
		}
	}

	return crc;
}

std::string Join_Choices(const std::vector<const char*>& choices)
//
//
// Join together a list of choices
//
//
{
	std::string join;

	for (const auto& walker : choices)
		join += std::string(!join.empty() ? ", " : "") + walker;

	if (choices.size() > 1)
		join.replace(join.rfind(", "), 2, " or ");

	return join;
}

void Open_File(FILE*& handle, const std::string& name, const char* mode)
//
//
// Open a file, failure is fatal
//
//
{
	if ((handle = fopen(name.c_str(), mode)) == NULL)
		Issue_Fatal_Error(MAKE_STRING("cannot open file '" << name << "'"));
}

void Open_FStream(std::fstream& stream, const std::string& name, std::ios_base::openmode mode)
//
//
// Open a file, failure is fatal
//
//
{
	stream.open(name.c_str(), mode);

	if (stream.fail())
		Issue_Fatal_Error(MAKE_STRING("cannot open file '" << name << "'"));
}

void Open_Card(FILE*& handle, const std::string& name, const char* mode)
//
//
// Open the flash card, failure is fatal
//
//
{
	if ((handle = FOPEN_64(name.c_str(), mode)) == NULL)
		Issue_Fatal_Error(MAKE_STRING("cannot access flash card at '" << name << "'"));
}

bool Size_Of_File(const std::string& file_name, nval32& file_size)
//
//
// Determine a file's size (file may exist)
//
// Returns true if file exists, else false
//
//
{
	try
	{
		file_size = static_cast<nval32>(std::filesystem::file_size(file_name));
	}
	catch (std::filesystem::filesystem_error&)
	{
		return false;
	}

	return true;
}

nval32 Size_Of_File(const std::string& file_name)
//
//
// Determine a file's size (file must exist)
//
//
{
	nval32 file_size;

	if (!Size_Of_File(file_name, file_size))
		Issue_Fatal_Error(MAKE_STRING("cannot find file '" << file_name << "'"));

	return file_size;
}

void Load_File(const std::string& file_name, nval8* file_buffer)
//
//
// Load an entire file
//
//
{
	nval32	file_size;
	FILE*	input_file;

	file_size = Size_Of_File(file_name);

	Open_File(input_file, file_name, "rb");

	fread(file_buffer, 1, file_size, input_file);

	fclose(input_file);
}

void Save_File(const std::string& file_name, const nval8* file_buffer, nval32 file_size)
//
//
// Save an entire file
//
//
{
	FILE* output_file;

	Open_File(output_file, file_name, "wb");

	fwrite(file_buffer, 1, file_size, output_file);

	fclose(output_file);
}

void Add_Script_Line(std::list<Script_Line>& script_lines, const std::string& line_text, bool first_line)
//
//
// Add a single script line
//
//
{
	static int	line_number = 0;
	static int	source_number = 0;

	if (first_line)
		line_number = 0;

	if (line_number == 0)
		source_number++;

	line_number++;

	script_lines.push_back(Script_Line(source_number, line_number, line_text));
}

void Add_Script_File(std::list<Script_Line>& script_lines, const std::string& file_name)
//
//
// Add a file's script lines
//
//
{
	char		ch;
	bool		first;
	char		terminator;
	std::string	input_line;
	FILE*		input_file;

	Open_File(input_file, file_name, "rb");

	first = true;

	terminator = '\0';

	while (!feof(input_file))
	{
		input_line = "";

		while (true)
		{
			ch = fgetc(input_file);

			if (ch == EOF || ch == '\n' || ch == '\r')
				break;

			input_line += ch;
		}

		if (terminator == '\0')
			terminator = ch;

		if (input_line.length() != 0 || ch == terminator)
		{
			Add_Script_Line(script_lines, input_line, first);

			first = false;
		}
	}

	fclose(input_file);
}

object_type_t Object_Type_Id(const Catalogue_Entry* cat_entry)
//
//
// Return 'id' of object type
//
//
{
	for (const auto& walker : Object_Type_Info)
		if (cat_entry->Type_Id() == walker.int_id)
			return walker.ext_id;

	return object_type_unknown;
}

std::string Object_Type_Description(const Catalogue_Entry* cat_entry, bool verbose = false)
//
//
// Return 'description' of object type
//
//
{
	std::ostringstream	description;
	const object_type_t	id = Object_Type_Id(cat_entry);

	if (verbose)
	{
		switch (id)
		{
		case object_type_fs :
			description << cat_entry->Stream("an FS #%std");
			break;
		case object_type_tool :
			description << cat_entry->Stream("a tool named '%nms'");
			break;
		case object_type_medium :
			description << cat_entry->Stream("a medium named '%nms'");
			break;
		case object_type_dfs_disc :
			description << cat_entry->Stream("a DFS disc named '%nms'");
			break;
		case object_type_adfs_disc :
			description << cat_entry->Stream("an ADFS disc named '%nms'");
			break;
		case object_type_hadfs_disc :
			description << cat_entry->Stream("a HADFS disc named '%nms'");
			break;
		case object_type_tape :
			description << cat_entry->Stream("a tape named '%nms'");
			break;
		case object_type_ue :
			description << cat_entry->Stream("an UE #%std");
			break;
		case object_type_file :
			description << cat_entry->Stream("a file named '%nms'");
			break;
		default :
			description << "an unknown object";
			break;
		}
	}
	else
	{
		switch (id)
		{
		case object_type_fs :
			description << "FS";
			break;
		case object_type_tool :
			description << "tool";
			break;
		case object_type_medium :
			description << "medium";
			break;
		case object_type_dfs_disc :
			description << "DFS disc";
			break;
		case object_type_adfs_disc :
			description << "ADFS disc";
			break;
		case object_type_hadfs_disc :
			description << "HADFS disc";
			break;
		case object_type_tape :
			description << "tape";
			break;
		case object_type_ue :
			description << "UE";
			break;
		case object_type_file :
			description << "file";
			break;
		default :
			description << "unknown object";
			break;
		}
	}

	return description.str();
}

int Recognize_FS(const std::string& file_name)
//
//
// Attempt to recognize a filing system
//
// Returns FS number if recognized, else 0
//
//
{
	nval32	full_crc;
	nval32	test_crc;
	nval8	file_data[16 * 1024];

	Load_File(file_name, file_data);

	full_crc = Crc_32(file_data, 16 * 1024);

	for (const auto& walker : Filing_System_Info)
	{
		test_crc = (walker.skip == 0) ? full_crc : Crc_32(file_data + walker.skip, walker.note);

		if (walker.crc_p != 0 && walker.crc_p == test_crc)
			return walker.number;
		if (walker.crc_o != 0 && walker.crc_o == test_crc)
			return 80 + walker.number;
	}

	return 0;
}

void Toggle_Disc_Interlacing(bool interlace, bool adfs, nval8* old_disc_data, nval32 old_disc_size, nval8* new_disc_data, nval32 new_disc_size)
//
//
// 'Toggle' disc interlacing
//
//
{
	const copy_map_entry*	map_walker;
	nval32			bytes_left;
	nval32			bytes_todo;
	nval32			conti_bytes;
	nval32			disco_bytes;
	nval32			conti_offset;
	nval32			disco_offset;
	nval8*			conti_data = interlace ? new_disc_data : old_disc_data;
	nval8*			disco_data = interlace ? old_disc_data : new_disc_data;
	nval32			conti_maxi = interlace ? new_disc_size : old_disc_size;
	nval32			disco_maxi = interlace ? old_disc_size : new_disc_size;
	const copy_map_entry	copy_map_dfs[]  = { { 10 * 256, 200 * 1024, 1 }, { 10 * 256, 10 * 256 - 200 * 1024, -1 } };
	const copy_map_entry	copy_map_adfs[] = { { 16 * 256, 320 * 1024, 1 }, { 16 * 256, 16 * 256 - 320 * 1024, -1 } };

	conti_offset = disco_offset = 0;

	bytes_left = adfs ? 640 * 1024 : 400 * 1024;

	map_walker = adfs ? copy_map_adfs : copy_map_dfs;

	memset(new_disc_data, FILLER_BYTE_DISC, new_disc_size);

	while (bytes_left > 0)
	{
		conti_bytes = (conti_offset > conti_maxi) ? 0 : conti_maxi - conti_offset;
		disco_bytes = (disco_offset > disco_maxi) ? 0 : disco_maxi - disco_offset;

		bytes_todo = std::min(map_walker->size, std::min(conti_bytes, disco_bytes));

		if (interlace)
			memcpy(conti_data + conti_offset, disco_data + disco_offset, bytes_todo);
		else
			memcpy(disco_data + disco_offset, conti_data + conti_offset, bytes_todo);

		conti_offset += map_walker->size;

		disco_offset += map_walker->step;

		bytes_left -= map_walker->size;

		map_walker += map_walker->jump;
	}
}

void Deinterlace_Disc(bool adfs, nval8* old_disc_data, nval32 old_disc_size, nval8* new_disc_data, nval32 new_disc_size)
//
//
// Deinterlace (A)DFS disc
//
//
{
	Toggle_Disc_Interlacing(false, adfs, old_disc_data, old_disc_size, new_disc_data, new_disc_size);
}

void Reinterlace_Disc(bool adfs, nval8* old_disc_data, nval32 old_disc_size, nval8* new_disc_data, nval32 new_disc_size)
//
//
// Reinterlace (A)DFS disc
//
//
{
	Toggle_Disc_Interlacing(true, adfs, old_disc_data, old_disc_size, new_disc_data, new_disc_size);
}

std::string Verbalise_Drive_Error(zeridajh::as_error error)
//
//
// 'Verbalise' drive error
//
//
{
	switch (error)
	{
	case zeridajh::as_error_drive_too_big :
		return "drive is bigger than maximum";
	case zeridajh::as_error_drive_too_small :
		return "drive is smaller than minimum";
	case zeridajh::as_error_drive_inconsistent :
		return "drive contents are inconsistent";
	case zeridajh::as_error_drive_data_missing :
		return "some drive data is missing";
	default :
		break;
	}

	return "unknown error";
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Catalogue_Entry
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Catalogue_Entry* Catalogue_Entry::Create(const nval8* raw)
{
	switch (raw[0])
	{
	case type_fs :
		return new Catalogue_Entry_FS();
	case type_tool :
		return new Catalogue_Entry_Tool();
	case type_medium :
		switch (raw[1])
		{
		case subtype_medium_dfs :
			return new Catalogue_Entry_DFS_Disc();
		case subtype_medium_adfs :
			return new Catalogue_Entry_ADFS_Disc();
		case subtype_medium_cfs :
			return new Catalogue_Entry_Tape();
		case subtype_medium_hadfs :
			return new Catalogue_Entry_HADFS_Disc();
		}
		break;
	case type_ue :
		return new Catalogue_Entry_UE();
	case type_file :
		return new Catalogue_Entry_File();
	}

	return new Catalogue_Entry_Unknown();
}

void Catalogue_Entry::Import(const nval8* raw)
{
	zeridajh::CData_Block data_block(raw);

	type = data_block.NVal8(0);
	subtype = data_block.NVal8(1);
	property = data_block.NVal8(2);
	card_address = data_block.NVal32(4);
	size_in_bytes = data_block.NVal32(8);

	name = data_block.String(16)(name_size_full).c_str(); // String can end early, at first NUL
}

void Catalogue_Entry::Export(nval8* raw) const
{
	zeridajh::CData_Block data_block(raw);

	data_block.Bytes(CATALOGUE_ENTRY_SIZE, 0) = (nval8) 0x00;

	data_block.NVal8(0) = type;
	data_block.NVal8(1) = subtype;
	data_block.NVal8(2) = property;
	data_block.NVal32(4) = card_address;
	data_block.NVal32(8) = size_in_bytes;

	data_block.String(16) = name;
}

nval16 Catalogue_Entry::Type_Id() const
{
	return (type == type_medium) ? type << 8 | subtype << 0 : type << 8;
}

bool Catalogue_Entry::Deleted() const
{
	return type == type_none || (property & property_deleted) != 0;
}

nval32 Catalogue_Entry::Card_Address() const
{
	return card_address;
}

nval32 Catalogue_Entry::Allocated_Bytes() const
{
	return SIZE_ON_CARD(size_in_bytes);
}

nval32 Catalogue_Entry::Populated_Bytes() const
{
	return size_in_bytes;
}

void Catalogue_Entry::Card_Address(nval32 a_card_address)
{
	card_address = a_card_address;
}

bool Catalogue_Entry::operator<(const Catalogue_Entry& other) const
{
	if (type < other.type)
		return true;

	if (type > other.type)
		return false;

	if (type == type_fs || type == type_ue)
	{
		if (subtype < other.subtype)
			return true;

		if (subtype > other.subtype)
			return false;
	}
	else
	{
		const int order = STRCASECMP(name.c_str(), other.name.c_str());

		if (order != 0)
			return (order < 0);
	}

	if ((property & property_deleted) != 0 && (other.property & property_deleted) == 0)
		return true;

	if ((property & property_deleted) == 0 && (other.property & property_deleted) != 0)
		return false;

	return false;
}

bool Catalogue_Entry::Compare_With(const Catalogue_Entry& other, bool clash_else_match) const
{
	if (Deleted())
		return false;

	if (type != other.type)
		return false;

	if (other.type == type_fs || other.type == type_ue)
	{
		if (subtype != other.subtype)
			return false;
	}
	else
	{
		if (STRCASECMP(name.c_str(), other.name.c_str()) != 0)
			return false;

		if (!clash_else_match && other.subtype != subtype_none && subtype != other.subtype)
			return false;
	}

	return true;
}

bool Catalogue_Entry::Matches_With(const Catalogue_Entry& other) const
{
	return Compare_With(other, false);
}

bool Catalogue_Entry::Clashes_With(const Catalogue_Entry& other) const
{
	return Compare_With(other, true);
}

void Catalogue_Entry::Delete()
{
	property |= property_deleted;
}

void Catalogue_Entry::Rename(const std::string& a_name)
{
	name = a_name;
}

std::string Catalogue_Entry::Stream(const std::string& format) const
{
	size_t		cursor;
	std::string	formatted = format;

	while ((cursor = formatted.find("%tyd")) != std::string::npos)
		formatted.replace(cursor, 4, std::to_string(type));

	while ((cursor = formatted.find("%tyh")) != std::string::npos)
		formatted.replace(cursor, 4, In_Hex(type, 2));

	while ((cursor = formatted.find("%std")) != std::string::npos)
		formatted.replace(cursor, 4, std::to_string(subtype));

	while ((cursor = formatted.find("%sth")) != std::string::npos)
		formatted.replace(cursor, 4, In_Hex(subtype, 2));

	while ((cursor = formatted.find("%nms")) != std::string::npos)
		formatted.replace(cursor, 4, name);

	while ((cursor = formatted.find("%prh")) != std::string::npos)
		formatted.replace(cursor, 4, In_Hex(property, 2));

	while ((cursor = formatted.find("%cah")) != std::string::npos)
		formatted.replace(cursor, 4, In_Hex(card_address, 8));

	while ((cursor = formatted.find("%abh")) != std::string::npos)
		formatted.replace(cursor, 4, In_Hex(Allocated_Bytes(), 8));

	while ((cursor = formatted.find("%pbh")) != std::string::npos)
		formatted.replace(cursor, 4, In_Hex(Populated_Bytes(), 8));

	while ((cursor = formatted.find("%vbh")) != std::string::npos)
		formatted.replace(cursor, 4, In_Hex(Allocated_Bytes() - Populated_Bytes(), 8));

	return formatted;
}

void Catalogue_Entry_File::Import(const nval8* raw)
{
	Catalogue_Entry::Import(raw);

	zeridajh::CData_Block data_block(raw);

	load_address = data_block.NVal16(54);
	exec_address = data_block.NVal16(57);
	current_size = data_block.NVal24(60);
}

void Catalogue_Entry_File::Export(nval8* raw) const
{
	Catalogue_Entry::Export(raw);

	zeridajh::CData_Block data_block(raw);

	data_block.NVal16(54) = load_address;
	data_block.NVal16(57) = exec_address;
	data_block.NVal24(60) = current_size;
}

nval32 Catalogue_Entry_File::Populated_Bytes() const
{
	return current_size;
}

std::string Catalogue_Entry_File::Stream(const std::string& format) const
{
	size_t		cursor;
	std::string	formatted = Catalogue_Entry::Stream(format);

	while ((cursor = formatted.find("%lah")) != std::string::npos)
		formatted.replace(cursor, 4, In_Hex(load_address, 4));

	while ((cursor = formatted.find("%eah")) != std::string::npos)
		formatted.replace(cursor, 4, In_Hex(exec_address, 4));

	return formatted;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Command
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Command::Command(Script_Line& a_script_line) : script_line(a_script_line)
{
	options = The_Options;

	old_entry = NULL;
	new_entry = NULL;
}

Command::~Command()
{
	if (old_entry != NULL)
		delete old_entry;
	if (new_entry != NULL)
		delete new_entry;
}

std::string Command::Arg(const char* what, bool left_else_pass)
//
//
// Return next argument (never empty)
//
//
{
	if (script_line.Empty())
		Issue_Fatal_Error(script_line, MAKE_STRING("expected " << what << ", got ''"));

	return left_else_pass ? script_line.Left() : script_line.Pass();
}

object_type_t Command::Arg_Object_Type()
//
//
// Return next argument (object type)
//
//
{
	std::vector<const char*> choices;

	for (const auto& walker : Object_Type_Info)
		choices.push_back(walker.name);

	return Object_Type_Info.at(Arg_Choice("object type", choices) - 1).ext_id;
}

Command::object_name_t Command::Arg_Object_Name(size_t limit)
//
//
// Return next argument (object name)
//
//
{
	const std::string arg = Arg("object name", false);

	if (arg.size() > limit)
		Issue_Fatal_Error(script_line, MAKE_STRING("object name is longer than " << limit << " characters"));

	for (int index = arg.size() - 1; index >= 0; index--)
		if (arg[index] < 33 || arg[index] > 126)
			Issue_Fatal_Error(script_line, "object name contains one or more illegal characters");

	return arg;
}

Command::object_number_t Command::Arg_Object_Number(const char* what)
//
//
// Return next argument (object number)
//
//
{
	char			dummy;
	unsigned long		value;
	const std::string	arg = Arg(what, false);

	if (sscanf(arg.c_str(), "%lu%c", &value, &dummy) != 1 || value > 255)
		Issue_Fatal_Error(script_line, MAKE_STRING(what << " should be between 0 and 255"));

	return static_cast<Command::object_number_t>(value);
}

Command::object_address_t Command::Arg_Object_Address(const char* what)
//
//
// Return next argument (object address)
//
//
{
	char			dummy;
	unsigned long		value;
	const std::string	arg = Arg(what, false);

	if (arg == "-")
		value = 0xBEADDADD;
	else if (sscanf(arg.c_str(), "%lx%c", &value, &dummy) != 1 || value > 0xFFFFFF)
		Issue_Fatal_Error(script_line, MAKE_STRING(what << " should be between 000000 and FFFFFF"));

	return static_cast<Command::object_address_t>(value);
}

Command::file_spec_t Command::Arg_File_Name(bool check)
//
//
// Return next argument (file name)
//
//
{
	file_spec_t		value;
	const std::string	arg = Arg("file name", true);

	value.name = arg;
	value.size = 0;

	if (check && !Size_Of_File(arg, value.size))
		Issue_Fatal_Error(script_line, MAKE_STRING("file '" << arg << "' does not exist"));

	return value;
}

std::string Command::Arg_Directory_Name()
//
//
// Return next argument (directory name)
//
//
{
	const std::string arg = Arg("directory name", true);

	return arg;
}

Command::disc_size_t Command::Arg_Disc_Size(const char* what)
//
//
// Return next argument (one or two disc sizes)
//
//
{
	char			dummy;
	unsigned long		size = 0;
	char			multy = '\0';
	const std::string	arg = Arg(what, false);

	if (sscanf(arg.c_str(), "%lu%c%c", &size, &multy, &dummy) != 2 || strchr("KM", multy) == NULL)
		Issue_Fatal_Error(script_line, MAKE_STRING(what << " is invalid, use e.g. '100K' or '20M'"));

	return size * ((multy == 'M') ? 1024 * 1024 : 1024);
}

Command::file_room_t Command::Arg_File_Room()
//
//
// Return next argument (file room)
//
//
{
	char			dummy;
	unsigned long		room = 0;
	char			multy = '\0';
	const std::string	arg = Arg("file room", false);

	if (sscanf(arg.c_str(), "%lu%c%c", &room, &multy, &dummy) != 2 || strchr("KM", multy) == NULL)
		Issue_Fatal_Error(script_line, "file room is invalid, use e.g. '1K' or '16M'");

	return room * ((multy == 'M') ? 1024 * 1024 : 1024);
}

int Command::Arg_Choice(const char* what, const std::vector<const char*>& choices)
//
//
// Return next argument (choice of literals)
//
//
{
	const int choice = script_line.Pass(choices);

	if (choice == 0)
		Issue_Fatal_Error(script_line, MAKE_STRING("expected " << what << " " << Join_Choices(choices) << ", got '" << script_line.Pass() << "'"));

	return choice;
}

void Command::Arg_Finish()
//
//
// Expect no further arguments
//
//
{
	if (!script_line.Empty())
		Issue_Fatal_Error(script_line, MAKE_STRING("expected nothing, got '" << script_line.Left() << "'"));
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Command_Do
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Command_Do::Command_Do(Script_Line& script_line) : Command(script_line)
{
	const int arg_option_name = Arg_Choice("option name", { "SHRINK", "BLOWUP", "COMPACT", "NOSHRINK", "NOBLOWUP", "NOCOMPACT", "NONE" });

	switch (arg_option_name)
	{
	case 1 : // SHRINK
		{
			The_Options.shrink = true;
		}
		break;
	case 2 : // BLOWUP
		{
			const disc_size_t arg_target_size = Arg_Disc_Size("target size");

			The_Options.blowup = true;

			The_Options.blowup_size = arg_target_size;
		}
		break;
	case 3 : // COMPACT
		{
			The_Options.compact = true;
		}
		break;
	case 4 : // NOSHRINK
		{
			The_Options.shrink = false;
		}
		break;
	case 5 : // NOBLOWUP
		{
			The_Options.blowup = false;
		}
		break;
	case 6 : // NOCOMPACT
		{
			The_Options.compact = false;
		}
		break;
	case 7 : // NONE
		{
			The_Options.shrink = false;
			The_Options.blowup = false;
			The_Options.compact = false;
		}
		break;
	}

	Arg_Finish();
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Command_Add
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Command_Add::Command_Add(Script_Line& script_line) : Command(script_line)
{
	const object_type_t arg_object_type = Arg_Object_Type();

	process_type = process_type_generic;

	switch (arg_object_type)
	{
	case object_type_fs :
		{
			const object_number_t	arg_fs_number = Arg_Object_Number("FS number");
			const file_spec_t	arg_file_name = Arg_File_Name(true);

			if (arg_file_name.size != 16 * 1024)
				Issue_Fatal_Error(script_line, "FS file size should be exactly 16K");

			int			fs_nr;
			const object_number_t	specified = arg_fs_number;
			const object_number_t	detected = Recognize_FS(arg_file_name.name);

			if (specified == 0)
			{
				if (detected == 0)
					Issue_Fatal_Error(script_line, "FS file not recognized as any known FS");

				fs_nr = detected;
			}
			else
			{
				if (VALID_FS_NUMBER(specified) || detected != 0)
					if (specified != detected)
						Issue_Fatal_Error(script_line, MAKE_STRING("FS number incorrect (specified " << specified << ", detected " << detected << ")"));

				fs_nr = specified;
			}

			source_file = arg_file_name.name;

			new_entry = new Catalogue_Entry_FS(fs_nr, FS_NUMBER_TO_NAME(fs_nr), arg_file_name.size);
		}
		break;
	case object_type_tool :
	case object_type_hadfs_disc :
		{
			const object_name_t	arg_object_name = Arg_Object_Name(Catalogue_Entry::name_size_full);
			const file_spec_t	arg_file_name = Arg_File_Name(true);

			source_file = arg_file_name.name;

			if (arg_object_type == object_type_tool)
				new_entry = new Catalogue_Entry_Tool(arg_object_name, arg_file_name.size);
			else
				new_entry = new Catalogue_Entry_HADFS_Disc(arg_object_name, arg_file_name.size);
		}
		break;
	case object_type_dfs_disc :
	case object_type_adfs_disc :
		{
			const int		arg_lacing_indicator = Arg_Choice("lacing indicator", { "N", "I" });
			const object_name_t	arg_object_name = Arg_Object_Name(Catalogue_Entry::name_size_full);
			const file_spec_t	arg_file_name = Arg_File_Name(true);

			options.deinterlace = (arg_lacing_indicator == 2);

			const bool adfs_else_dfs = (arg_object_type == object_type_adfs_disc);

			process_type = adfs_else_dfs ? process_type_adfs_disc : process_type_dfs_disc;

			nval32 new_disc_size;

			Process_Disc(arg_file_name.name, adfs_else_dfs, options, new_disc_size);

			source_file = arg_file_name.name;

			if (arg_object_type == object_type_dfs_disc)
				new_entry = new Catalogue_Entry_DFS_Disc(arg_object_name, new_disc_size);
			else
				new_entry = new Catalogue_Entry_ADFS_Disc(arg_object_name, new_disc_size);
		}
		break;
	case object_type_tape :
		{
			const object_name_t	arg_object_name = Arg_Object_Name(Catalogue_Entry::name_size_full);
			const file_spec_t	arg_file_name = Arg_File_Name(true);

			nval8* tape_data = new nval8[arg_file_name.size];

			Load_File(arg_file_name.name, tape_data);

			nval32 new_tape_size;

			if (strcmp((const char*) tape_data, "UEF File!") == 0)
			{
				// An actual unprocessed UEF file, to be decoded

				Process_Tape(arg_file_name.name, new_tape_size);

				process_type = process_type_tape_raw;
			}
			else if (*tape_data == 0x2a)
			{
				// An already processed UEF file, left 'as is'

				new_tape_size = arg_file_name.size;

				process_type = process_type_tape_cooked;
			}
			else if (tape_data[0] == 0x1f && tape_data[1] == 0x8b)
				Issue_Fatal_Error(script_line, "tape file should be 'ungzipped'");
			else
				Issue_Fatal_Error(script_line, "tape file contents not recognized");

			delete[] tape_data;

			source_file = arg_file_name.name;

			new_entry = new Catalogue_Entry_Tape(arg_object_name, new_tape_size);
		}
		break;
	case object_type_ue :
		{
			const object_number_t	arg_ue_number = Arg_Object_Number("UE number");
			const file_spec_t	arg_file_name = Arg_File_Name(true);

			if (arg_file_name.size != 4 * 1024)
				Issue_Fatal_Error(script_line, "UE file size should be exactly 4K");

			source_file = arg_file_name.name;

			new_entry = new Catalogue_Entry_UE(arg_ue_number, UE_NUMBER_TO_NAME(arg_ue_number), arg_file_name.size);
		}
		break;
	case object_type_file :
		{
			object_name_t		arg_object_name = Arg_Object_Name(Catalogue_Entry::name_size_file);
			object_address_t	arg_load_address = Arg_Object_Address("load address");
			object_address_t	arg_exec_address = Arg_Object_Address("exec address");
			const file_spec_t	arg_file_name = Arg_File_Name(true);

			source_skip = 0;

			if (arg_object_name == "-" || arg_load_address == 0xBEADDADD || arg_exec_address == 0xBEADDADD)
			{
				if (arg_file_name.size >= 22)
				{
					std::ifstream add_file(arg_file_name.name);

					if (add_file.good())
					{
						struct
						{
							char	name[16];
							nval16	load;
							nval16	exec;
							nval16	size;
						}
						atm_header;

						add_file.read((char*) &atm_header, 22);

						add_file.close();

						if ((arg_file_name.size - atm_header.size) == 22)
						{
							if (arg_object_name == "-")
								arg_object_name = std::string(atm_header.name, 16).c_str();

							if (arg_load_address == 0xBEADDADD)
								arg_load_address = atm_header.load;

							if (arg_exec_address == 0xBEADDADD)
								arg_exec_address = atm_header.exec;

							source_skip = 22;
						}
					}
				}

				if (arg_object_name == "-" || arg_load_address == 0xBEADDADD || arg_exec_address == 0xBEADDADD)
					Issue_Fatal_Error(script_line, "Cannot derive file's name, load address or exec address, information not found");
			}

			const nval32 cooked_size = arg_file_name.size - source_skip;

			if (cooked_size >= 16 * 1024 * 1024)
				Issue_Fatal_Error(script_line, "file size should not exceed 16M - 1");

			process_type = process_type_file_cooked;

			source_file = arg_file_name.name;

			new_entry = new Catalogue_Entry_File(arg_object_name, SIZE_ON_CARD(cooked_size), arg_load_address, arg_exec_address, cooked_size);
		}
		break;
	default :
		Issue_Fatal_Error(script_line, "object type for ADD may not be MEDIUM");
		break;
	}

	cat_entry = new_entry;

	Arg_Finish();
}

void Command_Add::Update()
{
	The_New_Catalogue->Check_Entry_Absent(cat_entry, script_line);

	The_New_Catalogue->Add_Entry(new_entry);

	new_entry = NULL; // Object handed over
}

void Command_Add::Execute()
{
	Object_Plan object_plan;

	std::cout << "Adding " << Object_Type_Description(cat_entry) << cat_entry->Stream(" '%nms'") << std::endl;

	switch (process_type)
	{
	case process_type_generic :
	case process_type_tape_cooked :
		{
			object_plan.Add_Part(Object_Part(source_file, cat_entry->Populated_Bytes()));
		}
		break;
	case process_type_dfs_disc :
	case process_type_adfs_disc :
		{
			const bool adfs_else_dfs = (process_type == process_type_adfs_disc);

			nval32	old_disc_size;
			nval32	new_disc_size;
			nval8*	new_disc_data;

			Process_Disc(source_file, adfs_else_dfs, options, old_disc_size, new_disc_size, new_disc_data);

			if (new_disc_size < old_disc_size)
				std::cout << "<shrunk from " << old_disc_size << " to " << new_disc_size << " bytes, " << (old_disc_size - new_disc_size) << " less>" << std::endl;
			else if (new_disc_size > old_disc_size)
				std::cout << "<blown up from " << old_disc_size << " to " << new_disc_size << " bytes, " << (new_disc_size - old_disc_size) << " more>" << std::endl;

			if (new_disc_data == NULL)
			{
				const nval32 file_size = Size_Of_File(source_file);

				object_plan.Add_Part(Object_Part(source_file, file_size));

				object_plan.Add_Part(Object_Part(FILLER_BYTE_DISC, cat_entry->Populated_Bytes() - file_size));
			}
			else
				object_plan.Add_Part(Object_Part(new_disc_data, new_disc_size));
		}
		break;
	case process_type_tape_raw :
		{
			nval32	new_tape_size;
			nval8*	new_tape_data;

			Process_Tape(source_file, new_tape_size, new_tape_data);

			object_plan.Add_Part(Object_Part(new_tape_data, new_tape_size));
		}
		break;
	case process_type_file_cooked :
		{
			object_plan.Add_Part(Object_Part(source_file, cat_entry->Populated_Bytes(), source_skip));
		}
		break;
	default :
		return;
	}

	The_New_Objects->Object_To_Card(*cat_entry, object_plan);
}

nval32 Command_Add::Size_On_Card() const
{
	return cat_entry->Allocated_Bytes();
}

void Command_Add::Process_Disc(const std::string& file_name, bool adfs_else_dfs, Option_List& options, nval32& old_disc_size, nval32& new_disc_size, nval8*& new_disc_data, bool test) const
//
//
// Attempt to 'process' a disc file
//
// Returns its new size and/or data
//
//
{
	nval8*	disc_data;
	nval32	disc_size;

	disc_data = NULL;

	disc_size = old_disc_size = ROUND_UP(Size_Of_File(file_name), 256);

	if (options.deinterlace)
	{
		const nval32	file_size = disc_size;
		const nval32	lace_size = adfs_else_dfs ? 640 * 1024 : 400 * 1024;

		if (file_size > lace_size)
			Issue_Fatal_Error(script_line, MAKE_STRING("disc file size should not exceed " << (lace_size / 1024) << "K"));

		nval8* file_data = new nval8[file_size];

		Load_File(file_name, file_data);

		nval8* lace_data = new nval8[lace_size];

		Deinterlace_Disc(adfs_else_dfs, file_data, file_size, lace_data, lace_size);

		delete[] file_data;

		disc_data = lace_data;

		disc_size = lace_size;
	}

	if (options.compact || options.shrink || options.blowup)
	{
		zeridajh::CFS_Drive*	drive;
		zeridajh::CFS_Drive*	drives[4];
		nval32			drive_size;
		nval32			drive_sizes[4];
		nval32			input_done = 0;
		int			drive_count = 0;
		const nval32		input_size = disc_size;
		zeridajh::as_error	drive_error = zeridajh::as_error_none;
		const nval32		drive_limit = adfs_else_dfs ? 512 * 1024 * 1024 : 200 * 1024;

		while (input_done < input_size && drive_count < 4 && drive_error == zeridajh::as_error_none)
		{
			if (adfs_else_dfs)
			{
				if (disc_data == NULL)
					drive = new zeridajh::CADFS_Drive(file_name, input_done, !test);
				else
					drive = new zeridajh::CADFS_Drive(disc_data, disc_size, input_done, !test);
			}
			else
			{
				if (disc_data == NULL)
					drive = new zeridajh::CDFS_Drive(file_name, input_done, !test);
				else
					drive = new zeridajh::CDFS_Drive(disc_data, disc_size, input_done, !test);
			}

			drive_error = drive->Ok();

			drives[drive_count] = drive;

			if (drive_error == zeridajh::as_error_none)
			{
				drive_size = drive->Size_In_Sectors() * 256;

				drive_sizes[drive_count] = drive_size;

				input_done += drive_size;
			}

			drive_count++;
		}

		if (drive_error == zeridajh::as_error_none)
		{
			disc_size = 0;

			for (int drive_index = 0; drive_index < drive_count; drive_index++)
			{
				drive = drives[drive_index];

				drive_size = drive_sizes[drive_index];

				bool compact = (options.compact && drive_size <= drive_limit);

				bool shrink = (options.shrink && drive_size <= drive_limit);

				bool blowup = (options.blowup && drive_size < options.blowup_size && options.blowup_size <= drive_limit);

				if (compact || shrink || blowup)
				{
					if (compact || shrink)
						drive->Compact();

					if (shrink || blowup)
						drive->Resize(blowup ? options.blowup_size / 256 : 0);

					drive_size = drive->Size_In_Sectors() * 256;

					drive_sizes[drive_index] = drive_size;
				}

				disc_size += drive_size;

				if (test)
					delete drive;
			}

			if (!test)
			{
				if (disc_data != NULL)
					delete[] disc_data;

				nval8* disc_walker = disc_data = new nval8[disc_size];

				for (int drive_index = 0; drive_index < drive_count; drive_index++)
				{
					drives[drive_index]->Read_All(disc_walker, FILLER_BYTE_DISC, NULL, NULL);

					disc_walker += drive_sizes[drive_index];

					delete drives[drive_index];
				}
			}
		}
		else
		{
			for (int drive_index = 0; drive_index < drive_count; drive_index++)
				delete drives[drive_index];

			options.compact = options.shrink = options.blowup = false;

			Issue_Warning(script_line, MAKE_STRING("drive processing ignored due to drive error '" << Verbalise_Drive_Error(drive_error) << "'"));
		}
	}

	if (test)
		if (disc_data != NULL)
			delete[] disc_data;

	new_disc_data = disc_data;

	new_disc_size = disc_size;
}

void Command_Add::Process_Disc(const std::string& file_name, bool adfs_else_dfs, Option_List& options, nval32& new_disc_size) const
//
//
// Attempt to 'process' a disc file (size phase)
//
// Returns its new size
//
//
{
	nval32	dummy_nval32;
	nval8*	dummy_nval8p;

	Process_Disc(file_name, adfs_else_dfs, options, dummy_nval32, new_disc_size, dummy_nval8p, true);
}

void Command_Add::Process_Disc(const std::string& file_name, bool adfs_else_dfs, Option_List& options, nval32& old_disc_size, nval32& new_disc_size, nval8*& new_disc_data) const
//
//
// Attempt to 'process' a disc file (data phase)
//
// Returns its new data
//
//
{
	Process_Disc(file_name, adfs_else_dfs, options, old_disc_size, new_disc_size, new_disc_data, false);
}

void Command_Add::Process_Tape(const std::string& file_name, nval32& new_tape_size, nval8*& new_tape_data, bool test) const
//
//
// Attempt to 'process' a tape file
//
// Returns its new size and/or data
//
//
{
	nval16			blk_code;
	nval32			blk_size;
	const nval8*		blk_data;
	zeridajh::CData_Block	data_block_r;
	zeridajh::CData_Block	data_block_w;
	nval32			old_tape_size;

	old_tape_size = Size_Of_File(file_name);

	new_tape_data = new nval8[old_tape_size];

	Load_File(file_name, new_tape_data);

	data_block_r.Reset(new_tape_data);
	data_block_w.Reset(new_tape_data);

	new_tape_size = 0;

	data_block_r.Cursor() = 10 + 2;

	while (data_block_r.Cursor() < (zval32) old_tape_size)
	{
		blk_code = data_block_r.NVal16();
		blk_size = data_block_r.NVal32();
		blk_data = data_block_r.Bytes(blk_size)();

		if (blk_code == 0x100 && *blk_data == 0x2a)
		{
			if (!test)
				data_block_w.Bytes(blk_size) = blk_data;

			new_tape_size += blk_size;
		}
	}

	if (!test)
		data_block_w.NVal8() = 0x7a;
	else
		delete[] new_tape_data;

	new_tape_size++;
}

void Command_Add::Process_Tape(const std::string& file_name, nval32& new_tape_size) const
//
//
// Attempt to 'process' a tape file (size phase)
//
// Returns its new size
//
//
{
	nval8* dummy_nval8p;

	Process_Tape(file_name, new_tape_size, dummy_nval8p, true);
}

void Command_Add::Process_Tape(const std::string& file_name, nval32& new_tape_size, nval8*& new_tape_data) const
//
//
// Attempt to 'process' a tape file (data phase)
//
// Returns its new data
//
//
{
	Process_Tape(file_name, new_tape_size, new_tape_data, false);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Command_New
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Command_New::Command_New(Script_Line& script_line) : Command(script_line)
{
	const object_type_t arg_object_type = Arg_Object_Type();

	switch (arg_object_type)
	{
	case object_type_dfs_disc :
	case object_type_adfs_disc :
		{
			const object_name_t	arg_object_name = Arg_Object_Name(Catalogue_Entry::name_size_full);
			const disc_size_t	arg_disc_size = Arg_Disc_Size("disc size");

			const bool adfs_else_dfs = (arg_object_type == object_type_adfs_disc);

			if (adfs_else_dfs)
			{
				if (arg_disc_size < 2 * 1024 || arg_disc_size > 512 * 1024 * 1024)
					Issue_Fatal_Error(script_line, "new ADFS disc size is limited to 2K - 512M");
			}
			else
			{
				if (arg_disc_size < 1 * 1024 || arg_disc_size > 400 * 1024)
					Issue_Fatal_Error(script_line, "new DFS disc size is limited to 1K - 400K");
			}

			process_type = adfs_else_dfs ? process_type_adfs_disc : process_type_dfs_disc;

			target_size = arg_disc_size;

			if (arg_object_type == object_type_dfs_disc)
				new_entry = new Catalogue_Entry_DFS_Disc(arg_object_name, arg_disc_size);
			else
				new_entry = new Catalogue_Entry_ADFS_Disc(arg_object_name, arg_disc_size);
		}
		break;
	case object_type_file :
		{
			const object_name_t	arg_object_name = Arg_Object_Name(Catalogue_Entry::name_size_file);
			const file_room_t	arg_file_room = Arg_File_Room();

			if (arg_file_room > 16 * 1024 * 1024)
				Issue_Fatal_Error(script_line, "file room should not exceed 16M");

			process_type = process_type_file_cooked;

			target_size = arg_file_room;

			new_entry = new Catalogue_Entry_File(arg_object_name, SIZE_ON_CARD(arg_file_room), 0, 0, 0);
		}
		break;
	default :
		Issue_Fatal_Error(script_line, "object type for NEW may only be DFS DISC, ADFS DISC or FILE");
		break;
	}

	cat_entry = new_entry;

	Arg_Finish();
}

void Command_New::Update()
{
	The_New_Catalogue->Check_Entry_Absent(cat_entry, script_line);

	The_New_Catalogue->Add_Entry(new_entry);

	new_entry = NULL; // Object handed over
}

void Command_New::Execute()
{
	Object_Plan object_plan;

	std::cout << "Creating " << Object_Type_Description(cat_entry) << cat_entry->Stream(" '%nms'") << std::endl;

	switch (process_type)
	{
	case process_type_dfs_disc :
		{
			nval32 disc_left = target_size;

			while (disc_left > 0)
			{
				nval8*			part_data;
				nval32			part_size;
				nval32			drive_total = 0;
				const nval32		drive_todo = std::min(disc_left, (nval32) 0x32000ul);
				zeridajh::CDFS_Drive	drive(drive_todo / 256, "", 0);

				for (int part = 1;; part++)
				{
					if ((part_data = drive.Read_Part(part, NULL, NULL, &part_size, NULL)) == NULL)
						break;

					object_plan.Add_Part(Object_Part(part_data, part_size));

					drive_total += part_size;
				}

				object_plan.Add_Part(Object_Part(FILLER_BYTE_DISC, drive_todo - drive_total));

				disc_left -= drive_todo;
			}
		}
		break;
	case process_type_adfs_disc :
		{
			nval8*			part_data;
			nval32			part_size;
			nval32			drive_total = 0;
			const nval32		drive_todo = target_size;
			const nval16		disc_identifier = (srand(static_cast<unsigned int>(time(0))), rand());
			zeridajh::CADFS_Drive	drive(drive_todo / 256, disc_identifier, 0);

			for (int part = 1;; part++)
			{
				if ((part_data = drive.Read_Part(part, NULL, NULL, &part_size, NULL)) == NULL)
					break;

				object_plan.Add_Part(Object_Part(part_data, part_size));

				drive_total += part_size;
			}

			object_plan.Add_Part(Object_Part(FILLER_BYTE_DISC, drive_todo - drive_total));
		}
		break;
	case process_type_file_cooked :
		{
			object_plan.Add_Part(Object_Part(FILLER_BYTE_FILE, target_size));
		}
		break;
	default :
		return;
	}

	The_New_Objects->Object_To_Card(*cat_entry, object_plan);
}

nval32 Command_New::Size_On_Card() const
{
	return cat_entry->Allocated_Bytes();
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Command_Delete
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Command_Delete::Command_Delete(Script_Line& script_line) : Command(script_line)
{
	const object_type_t arg_object_type = Arg_Object_Type();

	switch (arg_object_type)
	{
	case object_type_fs :
		{
			const object_number_t arg_fs_number = Arg_Object_Number("FS number");

			old_entry = new Catalogue_Entry_FS(arg_fs_number, FS_NUMBER_TO_NAME(arg_fs_number));
		}
		break;
	case object_type_tool :
	case object_type_medium :
	case object_type_dfs_disc :
	case object_type_adfs_disc :
	case object_type_tape :
	case object_type_hadfs_disc :
		{
			const object_name_t arg_object_name = Arg_Object_Name(Catalogue_Entry::name_size_full);

			switch (arg_object_type)
			{
			case object_type_tool :
				{
					old_entry = new Catalogue_Entry_Tool(arg_object_name);
				}
				break;
			case object_type_medium :
				{
					old_entry = new Catalogue_Entry_Medium(arg_object_name);
				}
				break;
			case object_type_dfs_disc :
				{
					old_entry = new Catalogue_Entry_DFS_Disc(arg_object_name);
				}
				break;
			case object_type_adfs_disc :
				{
					old_entry = new Catalogue_Entry_ADFS_Disc(arg_object_name);
				}
				break;
			case object_type_tape :
				{
					old_entry = new Catalogue_Entry_Tape(arg_object_name);
				}
				break;
			case object_type_hadfs_disc :
				{
					old_entry = new Catalogue_Entry_HADFS_Disc(arg_object_name);
				}
				break;
			default :
				break;
			}
		}
		break;
	case object_type_ue :
		{
			const object_number_t arg_ue_number = Arg_Object_Number("UE number");

			old_entry = new Catalogue_Entry_UE(arg_ue_number, UE_NUMBER_TO_NAME(arg_ue_number));
		}
		break;
	case object_type_file :
		{
			const object_name_t arg_object_name = Arg_Object_Name(Catalogue_Entry::name_size_file);

			old_entry = new Catalogue_Entry_File(arg_object_name);
		}
		break;
	default :
		break;
	}

	Arg_Finish();
}

void Command_Delete::Update()
{
	cat_entry = The_New_Catalogue->Check_Entry_Present(old_entry, script_line);
}

void Command_Delete::Execute()
{
	std::cout << "Deleting " << Object_Type_Description(cat_entry) << cat_entry->Stream(" '%nms'") << std::endl;

	The_New_Catalogue->Delete_Entry(cat_entry);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Command_Rename
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Command_Rename::Command_Rename(Script_Line& script_line) : Command(script_line)
{
	const object_type_t arg_object_type = Arg_Object_Type();

	switch (arg_object_type)
	{
	case object_type_tool :
	case object_type_medium :
	case object_type_dfs_disc :
	case object_type_adfs_disc :
	case object_type_tape :
	case object_type_hadfs_disc :
		{
			const object_name_t	arg_old_object_name = Arg_Object_Name(Catalogue_Entry::name_size_full);
			const object_name_t	arg_new_object_name = Arg_Object_Name(Catalogue_Entry::name_size_full);

			switch (arg_object_type)
			{
			case object_type_tool :
				{
					old_entry = new Catalogue_Entry_Tool(arg_old_object_name);
					new_entry = new Catalogue_Entry_Tool(arg_new_object_name);
				}
				break;
			case object_type_medium :
				{
					old_entry = new Catalogue_Entry_Medium(arg_old_object_name);
					new_entry = new Catalogue_Entry_Medium(arg_new_object_name);
				}
				break;
			case object_type_dfs_disc :
				{
					old_entry = new Catalogue_Entry_DFS_Disc(arg_old_object_name);
					new_entry = new Catalogue_Entry_DFS_Disc(arg_new_object_name);
				}
				break;
			case object_type_adfs_disc :
				{
					old_entry = new Catalogue_Entry_ADFS_Disc(arg_old_object_name);
					new_entry = new Catalogue_Entry_ADFS_Disc(arg_new_object_name);
				}
				break;
			case object_type_tape :
				{
					old_entry = new Catalogue_Entry_Tape(arg_old_object_name);
					new_entry = new Catalogue_Entry_Tape(arg_new_object_name);
				}
				break;
			case object_type_hadfs_disc :
				{
					old_entry = new Catalogue_Entry_HADFS_Disc(arg_old_object_name);
					new_entry = new Catalogue_Entry_HADFS_Disc(arg_new_object_name);
				}
				break;
			default :
				break;
			}
		}
		break;
	case object_type_file :
		{
			const object_name_t	arg_old_object_name = Arg_Object_Name(Catalogue_Entry::name_size_file);
			const object_name_t	arg_new_object_name = Arg_Object_Name(Catalogue_Entry::name_size_file);

			old_entry = new Catalogue_Entry_File(arg_old_object_name);
			new_entry = new Catalogue_Entry_File(arg_new_object_name);
		}
		break;
	default :
		Issue_Fatal_Error(script_line, "object type for RENAME may not be FS or UE");
		break;
	}

	target_name = new_entry->Stream("%nms");

	Arg_Finish();
}

void Command_Rename::Update()
{
	cat_entry = The_New_Catalogue->Check_Entry_Present(old_entry, script_line);
}

void Command_Rename::Execute()
{
	The_New_Catalogue->Check_Entry_Absent(new_entry, script_line);

	std::cout << "Renaming " << Object_Type_Description(cat_entry) << cat_entry->Stream(" '%nms' to '") << target_name << "'" << std::endl;

	The_New_Catalogue->Rename_Entry(cat_entry, target_name);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Command_Backup
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Command_Backup::Command_Backup(Script_Line& script_line) : Command(script_line)
{
	directory_name = Arg_Directory_Name();

	Arg_Finish();
}

void Command_Backup::Update()
{
	/* Nothing to do */
}

void Command_Backup::Execute()
{
	std::fstream script_file;

	std::filesystem::create_directory(directory_name);

	Open_FStream(script_file, directory_name + SYS_SLASH + "script", std::ios_base::out);

	nval32 backup_number = 0;

	The_Old_Catalogue->Scan([&] (const Catalogue_Entry* cat_entry)
	{
		bool			unknown_object = false;
		const bool		deleted_object = cat_entry->Deleted();
		const std::string	backup_name = MAKE_STRING(std::setfill('0') << std::setw(8) << backup_number++);

		if (!deleted_object)
		{
			switch (Object_Type_Id(cat_entry))
			{
			case object_type_fs :
				script_file << cat_entry->Stream("ADD FS %std ") << backup_name << "\n";
				break;
			case object_type_tool :
				script_file << cat_entry->Stream("ADD TOOL %nms ") << backup_name << "\n";
				break;
			case object_type_dfs_disc :
				script_file << cat_entry->Stream("ADD DFS DISC N %nms ") << backup_name << "\n";
				break;
			case object_type_adfs_disc :
				script_file << cat_entry->Stream("ADD ADFS DISC N %nms ") << backup_name << "\n";
				break;
			case object_type_tape :
				script_file << cat_entry->Stream("ADD TAPE %nms ") << backup_name << "\n";
				break;
			case object_type_hadfs_disc :
				script_file << cat_entry->Stream("ADD HADFS DISC %nms ") << backup_name << "\n";
				break;
			case object_type_ue :
				script_file << cat_entry->Stream("ADD UE %std ") << backup_name << "\n";
				break;
			case object_type_file :
				script_file << cat_entry->Stream("ADD FILE %nms %lah %eah ") << backup_name << "\n";
				break;
			default :
				unknown_object = true;
				break;
			}
		}

		if (!unknown_object && !deleted_object)
		{
			Object_Plan object_plan;

			std::cout << "Backing up " << Object_Type_Description(cat_entry) << cat_entry->Stream(" '%nms'") << std::endl;

			object_plan.Add_Part(Object_Part(directory_name + SYS_SLASH + backup_name, cat_entry->Populated_Bytes()));

			The_Storage->Read_Object(cat_entry->Card_Address(), cat_entry->Populated_Bytes(), object_plan);
		}
		else
			std::cout << "Ignoring " << (deleted_object ? "deleted" : "unknown") << cat_entry->Stream(" type %tyd object '%nms'") << std::endl;
	});

	script_file.close();
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Command_Extract
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Command_Extract::Command_Extract(Script_Line& script_line) : Command(script_line)
{
	const object_type_t arg_object_type = Arg_Object_Type();

	process_type = process_type_generic;

	switch (arg_object_type)
	{
	case object_type_fs :
		{
			const object_number_t arg_fs_number = Arg_Object_Number("FS number");

			old_entry = new Catalogue_Entry_FS(arg_fs_number, FS_NUMBER_TO_NAME(arg_fs_number));
		}
		break;
	case object_type_tool :
	case object_type_medium :
	case object_type_tape :
	case object_type_hadfs_disc :
		{
			const object_name_t arg_object_name = Arg_Object_Name(Catalogue_Entry::name_size_full);

			switch (arg_object_type)
			{
			case object_type_tool :
				{
					old_entry = new Catalogue_Entry_Tool(arg_object_name);
				}
				break;
			case object_type_medium :
				{
					old_entry = new Catalogue_Entry_Medium(arg_object_name);
				}
				break;
			case object_type_tape :
				{
					old_entry = new Catalogue_Entry_Tape(arg_object_name);
				}
				break;
			case object_type_hadfs_disc :
				{
					old_entry = new Catalogue_Entry_HADFS_Disc(arg_object_name);
				}
				break;
			default :
				break;
			}
		}
		break;
	case object_type_dfs_disc :
	case object_type_adfs_disc :
		{
			const int		arg_lacing_indicator = Arg_Choice("lacing indicator", { "N", "I" });
			const object_name_t	arg_object_name = Arg_Object_Name(Catalogue_Entry::name_size_full);

			options.reinterlace = (arg_lacing_indicator == 2);

			const bool adfs_else_dfs = (arg_object_type == object_type_adfs_disc);

			process_type = adfs_else_dfs ? process_type_adfs_disc : process_type_dfs_disc;

			if (arg_object_type == object_type_dfs_disc)
				old_entry = new Catalogue_Entry_DFS_Disc(arg_object_name);
			else
				old_entry = new Catalogue_Entry_ADFS_Disc(arg_object_name);
		}
		break;
	case object_type_ue :
		{
			const object_number_t arg_ue_number = Arg_Object_Number("UE number");

			old_entry = new Catalogue_Entry_UE(arg_ue_number, UE_NUMBER_TO_NAME(arg_ue_number));
		}
		break;
	case object_type_file :
		{
			const object_name_t arg_object_name = Arg_Object_Name(Catalogue_Entry::name_size_file);

			old_entry = new Catalogue_Entry_File(arg_object_name);
		}
		break;
	default :
		break;
	}

	const file_spec_t arg_file_name = Arg_File_Name(false);

	target_file = arg_file_name.name;

	Arg_Finish();
}

void Command_Extract::Update()
{
	cat_entry = The_Old_Catalogue->Check_Entry_Present(old_entry, script_line);
}

void Command_Extract::Execute()
{
	Object_Plan object_plan;

	std::cout << "Extracting " << Object_Type_Description(cat_entry) << cat_entry->Stream(" '%nms'") << std::endl;

	switch (process_type)
	{
	case process_type_generic :
		{
			const nval32 size_in_bytes = cat_entry->Populated_Bytes();

			object_plan.Add_Part(Object_Part(target_file, size_in_bytes));

			The_Storage->Read_Object(cat_entry->Card_Address(), size_in_bytes, object_plan);
		}
		break;
	case process_type_dfs_disc :
	case process_type_adfs_disc :
		{
			const bool adfs_else_dfs = (process_type == process_type_adfs_disc);

			if (options.reinterlace)
			{
				const nval32	old_size = cat_entry->Populated_Bytes();
				const nval32	new_size = cat_entry->Populated_Bytes();

				nval8* old_data = new nval8[old_size];

				object_plan.Add_Part(Object_Part(old_data, old_size));

				The_Storage->Read_Object(cat_entry->Card_Address(), old_size, object_plan);

				nval8* new_data = new nval8[new_size];

				Reinterlace_Disc(adfs_else_dfs, old_data, old_size, new_data, new_size);

				delete[] old_data;

				Save_File(target_file, new_data, new_size);

				delete[] new_data;
			}
			else
			{
				object_plan.Add_Part(Object_Part(target_file, cat_entry->Populated_Bytes()));

				The_Storage->Read_Object(cat_entry->Card_Address(), cat_entry->Populated_Bytes(), object_plan);
			}
		}
		break;
	default :
		return;
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Command_List
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Command_List::Command_List()
{
	nothing_to_do = false;

	nr_deletes = 0;
	nr_renames = 0;
	nr_derives = 0;

	new_objects_count = 0;
	new_objects_bytes = 0;
}

void Command_List::Append(Script_Line& script_line)
//
//
// Processing phase 1 : parse and register
//
//
{
	if (script_line.Ignore())
		return;

	const std::vector<const char*> choices { "DO", "ADD", "NEW", "DELETE", "RENAME", "BACKUP", "EXTRACT" };

	const int choice = script_line.Pass(choices);

	if (choice == 0)
		Issue_Fatal_Error(script_line, MAKE_STRING("expected command name " << Join_Choices(choices) << ", got '" << script_line.Pass() << "'"));

	Command* phased_command = NULL;

	switch (choice)
	{
	case 1 : // DO
		{
			Command_Do direct_command(script_line);
		}
		break;
	case 2 : // ADD
		{
			phased_command = new Command_Add(script_line);

			new_objects_count++;

			new_objects_bytes += phased_command->Size_On_Card() + OBJECT_HEADER_SIZE;
		}
		break;
	case 3 : // NEW
		{
			phased_command = new Command_New(script_line);

			new_objects_count++;

			new_objects_bytes += phased_command->Size_On_Card() + OBJECT_HEADER_SIZE;
		}
		break;
	case 4 : // DELETE
		{
			phased_command = new Command_Delete(script_line);

			nr_deletes++;
		}
		break;
	case 5 : // RENAME
		{
			phased_command = new Command_Rename(script_line);

			nr_renames++;
		}
		break;
	case 6 : // BACKUP
		{
			phased_command = new Command_Backup(script_line);

			nr_derives++;
		}
		break;
	case 7 : // EXTRACT
		{
			phased_command = new Command_Extract(script_line);

			nr_derives++;
		}
		break;
	}

	if (phased_command != NULL)
		command_list.push_back(phased_command);
}

void Command_List::Evaluate()
//
//
// All commands are in, evaluate them
//
//
{
	if ((new_objects_count + nr_deletes + nr_renames + nr_derives) == 0)
	{
		Issue_Warning("no commands to execute");

		nothing_to_do = true;
	}

	if (new_objects_count != 0 && nr_deletes + nr_renames + nr_derives != 0)
		Issue_Fatal_Error("please don't mix ADD and/or NEW with other commands");

	if (nr_derives != 0 && new_objects_count + nr_deletes + nr_renames != 0)
		Issue_Fatal_Error("please don't mix EXTRACT and/or BACKUP with other commands");

	if (nr_deletes != 0 && new_objects_count + nr_renames + nr_derives != 0)
		Issue_Fatal_Error("please don't mix DELETE with other commands");

	if (nr_renames != 0 && new_objects_count + nr_deletes + nr_derives != 0)
		Issue_Fatal_Error("please don't mix RENAME with other commands");
}

void Command_List::Update()
//
//
// Processing phase 2 : catalogue lookup/insert
//
//
{
	if (nothing_to_do)
		return;

	std::cout << "\n- Updating commands" << std::endl;

	for (const auto& walker : command_list)
		walker->Update();

	std::cout << "- Updating done" << std::endl;
}

void Command_List::Execute()
//
//
// Processing phase 3 : make changes/do transfer
//
//
{
	if (nothing_to_do)
		return;

	std::cout << "\n- Executing commands" << std::endl;

	for (const auto& walker : command_list)
		walker->Execute();

	std::cout << "- Executing done" << std::endl;
}

nval32 Command_List::Delta_Catalogue() const
//
//
// Return number of new catalogue entries
//
//
{
	return new_objects_count;
}

nval32 Command_List::Delta_Objects() const
//
//
// Return number of new object bytes
//
//
{
	return new_objects_bytes;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Card_Storage
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Card_Storage::Card_Storage(const std::string& card_spec, nval32 a_cache_size)
{
#ifdef WINDOWS_CW

	// On newer CygWins, \\.\<drive> is no longer writable
	// We now have to use the equivalent device file (eek)

	std::string used_spec = card_spec;

	{
		std::fstream		file;
		std::string		word;
		std::string		last;
		const std::string	find = card_spec + ":\\";

		Open_FStream(file, "/proc/partitions", std::ios_base::in);

		while (file.good())
		{
			last = word;

			file >> word;

			if (word == find)
				used_spec = "/dev/" + last;
		}
	}

#else

	const std::string& used_spec = card_spec;

#endif

	card_device = SYS_DEVFMT;
	card_device.replace(card_device.find("%s"), 2, used_spec);

	if (a_cache_size == 0)
	{
		cache_size = 0;

		cache_buffer = NULL;
	}
	else
	{
		cache_size = SIZE_ON_CARD(a_cache_size);

		cache_buffer = new nval8[cache_size];
	}

	cache_fill_lo = cache_fill_hi = 0;
}

nval32 Card_Storage::Read(FILE* io_file, nval32& card_address, nval8*& data_buffer, nval32 bytes_to_transfer)
//
//
// Raw read from flash card (chunk)
//
//
{
	nval32	bytes_todo;
	nval32	bytes_done;
	nval32	address_skew;
	nval32	bytes_transferred;
	nval8	sector_buffer[CARD_SECTOR_SIZE];

#ifdef VERBOSE
	std::cout << "   sub-read " << std::hex << std::setw(8) << card_address << " + " << std::setw(8) << bytes_to_transfer << std::dec << std::endl;
#endif

	address_skew = card_address % CARD_SECTOR_SIZE;

	if (bytes_to_transfer < CARD_SECTOR_SIZE)
	{
		bytes_todo = CARD_SECTOR_SIZE;

		bytes_done = fread(sector_buffer, 1, bytes_todo, io_file);

		memcpy(data_buffer, &sector_buffer[address_skew], bytes_to_transfer);

		bytes_transferred = bytes_to_transfer;
	}
	else
	{
		bytes_todo = bytes_to_transfer;

		if (address_skew != 0)
			Issue_Fatal_Error(MAKE_STRING("unaligned read from flash card @ " << std::hex << std::setw(8) << card_address << " + " << std::setw(8) << bytes_todo));

		bytes_done = fread(data_buffer, 1, bytes_todo, io_file);

		bytes_transferred = bytes_done;
	}

	if (bytes_done < bytes_todo)
		if (bytes_todo <= CARD_SECTOR_SIZE)
			Issue_Fatal_Error(MAKE_STRING("failed to read from flash card @ " << std::hex << std::setw(8) << card_address << " + " << std::setw(8) << bytes_todo));

	card_address += bytes_transferred;

	data_buffer += bytes_transferred;

	return bytes_transferred;
}

void Card_Storage::Read(nval32 card_address, nval8* data_buffer, nval32 bytes_to_transfer)
//
//
// Raw read from flash card
//
//
{
	FILE*	io_file;
	nval32	lead_bytes;
	nval32	tail_bytes;

#ifdef VERBOSE
	std::cout << std::endl << "!! card read " << std::hex << std::setw(8) << card_address << " + " << std::setw(8) << bytes_to_transfer << std::dec << std::endl;
#endif

	if (bytes_to_transfer == 0)
		return;

	if ((lead_bytes = card_address % CARD_SECTOR_SIZE) != 0)
		lead_bytes = std::min(CARD_SECTOR_SIZE - lead_bytes, bytes_to_transfer);

	bytes_to_transfer -= lead_bytes;

	tail_bytes = bytes_to_transfer % CARD_SECTOR_SIZE;

	bytes_to_transfer -= tail_bytes;

	Open_Card(io_file, card_device, "rb");

	FSEEK_64(io_file, The_Area + ROUND_DOWN(card_address, CARD_SECTOR_SIZE), SEEK_SET);

	if (lead_bytes > 0)
		Read(io_file, card_address, data_buffer, lead_bytes);

	if (bytes_to_transfer > 0)
	{
		bytes_to_transfer -= Read(io_file, card_address, data_buffer, bytes_to_transfer);

		while (bytes_to_transfer > 0)
		{
			// There may be trouble reading near the end of the flash card,
			// even if not going 'over the edge', so the code reverts to
			// 'nibble' mode if the bigger fread (partially) fails.

			nval32 bytes_to_nibble = std::min((nval32) CARD_NIBBLE_SIZE, bytes_to_transfer);

#ifdef VERBOSE
			std::cout << "!! nibble !" << std::endl;
#endif

			bytes_to_transfer -= Read(io_file, card_address, data_buffer, bytes_to_nibble);
		}
	}

	if (tail_bytes > 0)
		Read(io_file, card_address, data_buffer, tail_bytes);

	fclose(io_file);

#ifdef VERBOSE
	std::cout << std::endl;
#endif
}

nval32 Card_Storage::Write(FILE* io_file, nval32& card_address, const nval8*& data_buffer, nval32 bytes_to_transfer)
//
//
// Raw write to flash card (chunk)
//
//
{
	nval32	bytes_todo;
	nval32	bytes_done;
	nval32	address_skew;
	nval32	bytes_transferred;
	nval8	sector_buffer[CARD_SECTOR_SIZE];

#ifdef VERBOSE
	std::cout << "   sub-write " << std::hex << std::setw(8) << card_address << " + " << std::setw(8) << bytes_to_transfer << std::dec << std::endl;
#endif

	address_skew = card_address % CARD_SECTOR_SIZE;

	if (bytes_to_transfer < CARD_SECTOR_SIZE)
	{
		bytes_todo = CARD_SECTOR_SIZE;

		fflush(io_file); // We must, write <-> read transitions need flush or seek

		FSEEK_64(io_file, -((OFFT_64) fread(sector_buffer, 1, bytes_todo, io_file)), SEEK_CUR);

		memcpy(&sector_buffer[address_skew], data_buffer, bytes_to_transfer);

		bytes_done = fwrite(sector_buffer, 1, bytes_todo, io_file);

		bytes_transferred = bytes_to_transfer;
	}
	else
	{
		bytes_todo = bytes_to_transfer;

		if (address_skew != 0)
			Issue_Fatal_Error(MAKE_STRING("unaligned write to flash card @ " << std::hex << std::setw(8) << card_address << " + " << std::setw(8) << bytes_todo));

		bytes_done = fwrite(data_buffer, 1, bytes_todo, io_file);

		bytes_transferred = bytes_done;
	}

	if (bytes_done < bytes_todo)
		Issue_Fatal_Error(MAKE_STRING("failed to write to flash card @ " << std::hex << std::setw(8) << card_address << " + " << std::setw(8) << bytes_todo));

	card_address += bytes_transferred;

	data_buffer += bytes_transferred;

	return bytes_transferred;
}

void Card_Storage::Write(nval32 card_address, const nval8* data_buffer, nval32 bytes_to_transfer)
//
//
// Raw write to flash card
//
//
{
	FILE*	io_file;
	nval32	lead_bytes;
	nval32	tail_bytes;

#ifdef VERBOSE
	std::cout << std::endl << "!! card write " << std::hex << std::setw(8) << card_address << " + " << std::setw(8) << bytes_to_transfer << std::dec << std::endl;
#endif

	if (bytes_to_transfer == 0)
		return;

	if ((lead_bytes = card_address % CARD_SECTOR_SIZE) != 0)
		lead_bytes = std::min(CARD_SECTOR_SIZE - lead_bytes, bytes_to_transfer);

	bytes_to_transfer -= lead_bytes;

	tail_bytes = bytes_to_transfer % CARD_SECTOR_SIZE;

	bytes_to_transfer -= tail_bytes;

	Open_Card(io_file, card_device, "rb+");

	FSEEK_64(io_file, The_Area + ROUND_DOWN(card_address, CARD_SECTOR_SIZE), SEEK_SET);

	if (lead_bytes > 0)
		Write(io_file, card_address, data_buffer, lead_bytes);

	if (bytes_to_transfer > 0)
		Write(io_file, card_address, data_buffer, bytes_to_transfer);

	if (tail_bytes > 0)
		Write(io_file, card_address, data_buffer, tail_bytes);

	fclose(io_file);

#ifdef VERBOSE
	std::cout << std::endl;
#endif
}

nval8* Card_Storage::Cache(nval32 card_address, nval32 bytes_to_cache)
//
//
// Cache write to flash card (data post-copied)
//
//
{
	nval8* reserved;

#ifdef VERBOSE
	std::cout << std::endl << "!! card cache " << std::hex << std::setw(8) << card_address << " + " << std::setw(8) << bytes_to_cache << std::dec << std::endl;
#endif

	if (bytes_to_cache == 0)
		return NULL;

	if (bytes_to_cache >= cache_size)
		return NULL;

	bool need_flush = true;

	if (bytes_to_cache <= cache_size - (cache_fill_hi - cache_fill_lo))
	{
		// Cache can accommodate, we flush only if we can't merge cached + incoming into one block

		if (cache_fill_lo == cache_fill_hi) // Cache is empty
			need_flush = false;
		else if (card_address == cache_card_address + cache_fill_hi) // Seamless with top of cached data
			need_flush = false;
		else if (card_address + bytes_to_cache == cache_card_address + cache_fill_lo) // Seamless with bottom of cached data
			need_flush = false;
	}

	if (need_flush)
		Flush();

	if (cache_fill_lo == cache_fill_hi)
		cache_card_address = card_address;

	if (card_address == cache_card_address + cache_fill_hi)
	{
		if (cache_size - cache_fill_hi < bytes_to_cache)
		{
			nval32 delta = cache_fill_lo;

			memmove(cache_buffer + cache_fill_lo - delta, cache_buffer + cache_fill_lo, cache_fill_hi - cache_fill_lo);

			cache_fill_hi -= delta;

			cache_fill_lo -= delta;

			cache_card_address += delta;

#ifdef VERBOSE
			std::cout << "   scroll down " << std::hex << std::setw(8) << delta << std::dec << " to make top room" << std::endl;
#endif
		}

		reserved = cache_buffer + cache_fill_hi;

		cache_fill_hi += bytes_to_cache;
	}
	else
	{
		if (cache_fill_lo < bytes_to_cache)
		{
			nval32 delta = cache_size - cache_fill_hi;

			memmove(cache_buffer + cache_fill_lo + delta, cache_buffer + cache_fill_lo, cache_fill_hi - cache_fill_lo);

			cache_fill_hi += delta;

			cache_fill_lo += delta;

			cache_card_address -= delta;

#ifdef VERBOSE
			std::cout << "   scroll up " << std::hex << std::setw(8) << delta << std::dec << " to make bottom room" << std::endl;
#endif
		}

		cache_fill_lo -= bytes_to_cache;

		reserved = cache_buffer + cache_fill_lo;
	}

#ifdef VERBOSE
	std::cout << std::endl;
#endif

	return reserved;
}

void Card_Storage::Flush()
//
//
// Flush out cached data
//
//
{
	const nval8*	buffer;
	nval32		card_address;
	nval32		bytes_to_write;

#ifdef VERBOSE
	std::cout << "\n!! flush" << std::endl;
#endif

	if ((bytes_to_write = cache_fill_hi - cache_fill_lo) == 0)
		return;

	buffer = cache_buffer + cache_fill_lo;

	card_address = cache_card_address + cache_fill_lo;

	Write(card_address, buffer, bytes_to_write);

	cache_fill_lo = cache_fill_hi = 0;
}

nval8* Card_Storage::Write_Cached(nval32 card_address, const nval8* buffer, nval32 bytes_to_write)
//
//
// Cached write to flash card
//
//
{
	nval8* reserved = Cache(card_address, bytes_to_write);

	if (buffer == NULL)
		return reserved;

	if (reserved != NULL)
		memcpy(reserved, buffer, bytes_to_write);
	else
		Write(card_address, buffer, bytes_to_write);

	return NULL;
}

void Card_Storage::Read_Object(nval32 card_address, nval32 size_in_bytes, const Object_Plan& object_plan)
//
//
// Read object from flash card
//
//
{
	nval32	part_bytes;
	nval32	part_address = card_address;
	nval32	object_bytes = size_in_bytes;

	for (std::list<Object_Part>::const_iterator current_part = object_plan.parts.begin(); current_part != object_plan.parts.end(); current_part++)
	{
		part_bytes = std::min(object_bytes, current_part->size);

		if (part_bytes == 0)
			continue;

		switch (current_part->type)
		{
		case objecr_part_file :
			{
				nval32	bytes_todo;
				FILE*	output_file;
				nval8*	data_buffer;
				nval32	bytes_left = part_bytes;
				nval32	card_walker = part_address;

				data_buffer = new nval8[CHUNK_BUFFER_SIZE];

				Open_File(output_file, current_part->name, "wb");

				while (bytes_left > 0)
				{
					bytes_todo = std::min(bytes_left, (nval32) CHUNK_BUFFER_SIZE);

					Read(card_walker, data_buffer, bytes_todo);

					fwrite(data_buffer, 1, bytes_todo, output_file);

					card_walker += bytes_todo;

					bytes_left -= bytes_todo;
				}

				fclose(output_file);

				delete[] data_buffer;
			}
			break;
		case objecr_part_data :
			{
				Read(part_address, current_part->data, part_bytes);
			}
			break;
		case objecr_part_made :
			/* Not used at present */
			break;
		}

		part_address += part_bytes;

		object_bytes -= part_bytes;
	}
}

void Card_Storage::Write_Object(nval32 card_address, nval32 size_in_bytes, const Object_Plan& object_plan)
//
//
// Write object to flash card
//
//
{
	bool	flush;
	nval32	part_bytes;
	nval8*	data_buffer;
	nval32	buffer_size;
	nval32	part_address = card_address;
	nval32	object_bytes = SIZE_ON_CARD(size_in_bytes);

	flush = false;

	buffer_size = object_bytes;

	data_buffer = Write_Cached(card_address, NULL, buffer_size);

	if (data_buffer == NULL)
	{
		flush = true;

		buffer_size = CHUNK_BUFFER_SIZE;

		data_buffer = new nval8[buffer_size];
	}

	for (std::list<Object_Part>::const_iterator current_part = object_plan.parts.begin(); current_part != object_plan.parts.end(); current_part++)
	{
		part_bytes = std::min(object_bytes, current_part->size);

		if (part_bytes == 0)
			continue;

		nval32		bytes_todo;
		FILE*		input_file = NULL;
		const nval8*	data_walker = NULL;
		nval32		bytes_left = part_bytes;
		nval32		card_walker = part_address;

		switch (current_part->type)
		{
		case objecr_part_file :
			Open_File(input_file, current_part->name, "rb");
			fseek(input_file, current_part->skip, SEEK_SET);
			break;
		case objecr_part_data :
			data_walker = current_part->data;
			break;
		default :
			break;
		}

		while (bytes_left > 0)
		{
			bytes_todo = std::min(bytes_left, buffer_size);

			switch (current_part->type)
			{
			case objecr_part_file :
				fread(data_buffer, 1, bytes_todo, input_file);
				break;
			case objecr_part_data :
				memcpy(data_buffer, data_walker, bytes_todo);
				data_walker += bytes_todo;
				break;
			case objecr_part_made :
				memset(data_buffer, current_part->byte, bytes_todo);
				break;
			}

			if (flush)
				Write(card_walker, data_buffer, bytes_todo);
			else
				data_buffer += bytes_todo;

			card_walker += bytes_todo;

			bytes_left -= bytes_todo;
		}

		switch (current_part->type)
		{
		case objecr_part_file :
			fclose(input_file);
			break;
		default :
			break;
		}

		part_address += part_bytes;

		object_bytes -= part_bytes;
	}

	part_bytes = object_bytes;

	if (part_bytes > 0)
	{
		memset(data_buffer, FILLER_BYTE_CARD, part_bytes);

		if (flush)
			Write(part_address, data_buffer, part_bytes);
	}

	if (flush)
		delete[] data_buffer;
}

void Card_Storage::Import_Data(nval32 card_address, nval8* buffer, nval32 bytes_to_read)
//
//
// 'Import' data from flash card
//
//
{
	Read(card_address, buffer, bytes_to_read);
}

void Card_Storage::Export_Data(nval32 card_address, const nval8* buffer, nval32 bytes_to_write)
//
//
// 'Export' (imported) data to flash card
//
//
{
	Write(card_address, buffer, bytes_to_write);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Card_Global_Header
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Card_Global_Header::Card_Global_Header(nval32 a_card_address, nval32 a_size_in_bytes, bool make_up_header) : card_address(a_card_address), size_in_bytes(a_size_in_bytes)
{
	//
	//
	// Initialise members
	//
	//

	dirty = false;

	//
	//
	// Read from flash card
	//
	//

	nval8* raw = new nval8[size_in_bytes];

	The_Storage->Import_Data(card_address, raw, size_in_bytes);

	contents.Import(raw);

	delete[] raw;

	if (contents.signature == CRD_SIGN_GHD)
		return;

	if (!make_up_header)
		Issue_Fatal_Error("Flash card global header has wrong signature");

	Issue_Warning("Flash card global header has wrong signature");

	std::cout << "\n!! Making up a header for a (roughly) 1 MB MMC due to '-x' !!"
		"\n!! Don't forget to reformat with MMCFO before any real use !!"
		<< std::endl
		;

	contents.signature = CRD_SIGN_GHD;

	contents.cat_free_ptr = 0;

	contents.dat_free_ptr = 1 * 1024 * 1024;

	dirty = true;
}

void Card_Global_Header::Flush()
//
//
// Write (back) to flash card
//
//
{
	if (!dirty)
		return;

	std::cout << "\n- Flushing flash card global header" << std::endl;

	nval8* raw = new nval8[size_in_bytes];

	contents.Export(raw);

	The_Storage->Export_Data(card_address, raw, size_in_bytes);

	delete[] raw;

	std::cout << "- Flushing done" << std::endl;

	dirty = false;
}

nval32 Card_Global_Header::Catalogue_Entries() const
//
//
// Return number of catalogue entries
//
//
{
	return contents.cat_free_ptr;
}

nval32 Card_Global_Header::Free_Data_Pointer() const
//
//
// Return free data pointer
//
//
{
	return contents.dat_free_ptr;
}

void Card_Global_Header::Catalogue_Entries(nval32 new_value)
//
//
// Change number of catalogue entries
//
//
{
	if (contents.cat_free_ptr == new_value)
		return;

	contents.cat_free_ptr = new_value;

	dirty = true;
}

void Card_Global_Header::Free_Data_Pointer(nval32 new_value)
//
//
// Change free data pointer
//
//
{
	if (contents.dat_free_ptr == new_value)
		return;

	contents.dat_free_ptr = new_value;

	dirty = true;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Card_Catalogue
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Card_Catalogue::Card_Catalogue(nval32 a_card_address, nval32 size_in_entries) : card_address(a_card_address), current_entries(size_in_entries)
{
	//
	//
	// Initialise members
	//
	//

	dirty = false;

	//
	//
	// Read from flash card
	//
	//

	nval8* raw = new nval8[current_entries * CATALOGUE_ENTRY_SIZE];

	The_Storage->Import_Data(card_address, raw, current_entries * CATALOGUE_ENTRY_SIZE);

	contents.reserve(current_entries);

	for (nval32 index = 0; index < current_entries; index++)
	{
		contents.push_back(Catalogue_Entry::Create(raw + index * CATALOGUE_ENTRY_SIZE));

		contents.back()->Import(raw + index * CATALOGUE_ENTRY_SIZE);
	}

	delete[] raw;

	if (current_entries == 0)
	{
		objects_base = The_Global_Header->Free_Data_Pointer();
		objects_size = 0;
	}
	else
	{
		nval32	entry_lo;
		nval32	entry_hi;
		nval32	dat_lo = 0xffffffff;
		nval32	dat_hi = 0x00000000;

		for (auto cat_entry : contents)
		{
			entry_lo = cat_entry->Card_Address();
			entry_hi = entry_lo + cat_entry->Allocated_Bytes();

			if (entry_lo < dat_lo)
				dat_lo = entry_lo;

			if (entry_hi > dat_hi)
				dat_hi = entry_hi;
		}

		objects_base = dat_lo - OBJECT_HEADER_SIZE;
		objects_size = dat_hi - objects_base;
	}

	if (objects_base != The_Global_Header->Free_Data_Pointer())
		Issue_Fatal_Error("Flash card catalogue is inconsistent with free pointer");

	free_pointer = objects_base;
}

Card_Catalogue::~Card_Catalogue()
{
	for (auto cat_entry : contents)
		delete cat_entry;
}

void Card_Catalogue::Flush()
//
//
// Write (back) to flash card
//
//
{
	if (!dirty)
		return;

	std::cout << "\n- Flushing flash card catalogue" << std::endl;

	std::sort(contents.begin(), contents.end(), [] (const Catalogue_Entry* a, const Catalogue_Entry* b) { return *a < *b; });

	nval8* raw = new nval8[current_entries * CATALOGUE_ENTRY_SIZE];

	for (nval32 index = 0; index < current_entries; index++)
		contents.at(index)->Export(raw + index * CATALOGUE_ENTRY_SIZE);

	The_Storage->Export_Data(card_address, raw, current_entries * CATALOGUE_ENTRY_SIZE);

	delete[] raw;

	std::cout << "- Flushing done" << std::endl;

	dirty = false;
}

nval32 Card_Catalogue::Base_Address() const
//
//
// Return base address of (new) catalogue
//
//
{
	return card_address;
}

nval32 Card_Catalogue::End_Address() const
//
//
// Return end address of (new) catalogue
//
//
{
	return card_address + Size_In_Bytes();
}

nval32 Card_Catalogue::Size_In_Bytes() const
//
//
// Return size in bytes of (new) catalogue
//
//
{
	return Size_In_Entries() * CATALOGUE_ENTRY_SIZE;
}

nval32 Card_Catalogue::Size_In_Entries() const
//
//
// Return size in entries of (new) catalogue
//
//
{
	return current_entries;
}

nval32 Card_Catalogue::Objects_Base_Address() const
//
//
// Return base address of (old) objects
//
//
{
	return objects_base;
}

nval32 Card_Catalogue::Objects_End_Address() const
//
//
// Return end address of (old) objects
//
//
{
	return objects_base + objects_size;
}

nval32 Card_Catalogue::Objects_Size_In_Bytes() const
//
//
// Return size in bytes of (old) objects
//
//
{
	return objects_size;
}

void Card_Catalogue::Scan(std::function<void(const Catalogue_Entry*)> processor) const
//
//
// Call function for all catalogue entries
//
//
{
	for (auto cat_entry : contents)
		processor(cat_entry);
}

void Card_Catalogue::List() const
//
//
// List all entries
//
//
{
	std::cout << "\nCatalogue listing :\n" << std::endl;

	for (auto cat_entry : contents)
		std::cout << cat_entry->Stream("%tyh %sth %prh %cah + %abh (%pbh used + %vbh idle) %nms") << std::endl;

	if (current_entries == 0)
		std::cout << "<no entries>" << std::endl;
}

const Catalogue_Entry* Card_Catalogue::Check_Entry_Present(const Catalogue_Entry* old_entry, const Script_Line& script_line) const
//
//
// Check that an entry is present, else fatal error
//
// Return pointer to entry
//
//
{
	for (auto cat_entry : contents)
		if (cat_entry->Matches_With(*old_entry))
			return cat_entry;

	Issue_Fatal_Error(script_line, MAKE_STRING(Object_Type_Description(old_entry, true) << " does not exist"));

	return NULL;
}

void Card_Catalogue::Check_Entry_Absent(const Catalogue_Entry* new_entry, const Script_Line& script_line) const
//
//
// Check that an entry is absent, else fatal error
//
//
{
	for (auto cat_entry : contents)
		if (cat_entry->Clashes_With(*new_entry))
			Issue_Fatal_Error(script_line, MAKE_STRING(Object_Type_Description(new_entry, true) << " already exists"));
}

void Card_Catalogue::Add_Entry(Catalogue_Entry* new_entry)
//
//
// Add a new entry
//
// Return pointer to new entry
//
//
{
	contents.push_back(new_entry);

	free_pointer -= new_entry->Allocated_Bytes();

	new_entry->Card_Address(free_pointer);

	free_pointer -= OBJECT_HEADER_SIZE;

	current_entries++;

	dirty = true;
}

void Card_Catalogue::Delete_Entry(const Catalogue_Entry* cat_entry)
//
//
// Delete an entry
//
//
{
	const_cast<Catalogue_Entry*>(cat_entry)->Delete();

	dirty = true;
}

void Card_Catalogue::Rename_Entry(const Catalogue_Entry* cat_entry, const std::string& new_name)
//
//
// Rename an entry
//
//
{
	const_cast<Catalogue_Entry*>(cat_entry)->Rename(new_name);

	dirty = true;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Class Card_Objects
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Card_Objects::Card_Objects(nval32 bytes_to_add) : additional_bytes(bytes_to_add)
{
	dirty = false;

	card_address = The_Old_Catalogue->Objects_Base_Address();

	existing_bytes = The_Old_Catalogue->Objects_Size_In_Bytes();
}

void Card_Objects::Flush()
//
//
// Flush out new objects
//
//
{
	if (!dirty)
		return;

	The_Storage->Flush();

	dirty = false;
}

nval32 Card_Objects::Base_Address() const
//
//
// Return base address of (new) objects
//
//
{
	return (additional_bytes > card_address) ? 0 : card_address - additional_bytes;
}

nval32 Card_Objects::Size_In_Bytes() const
//
//
// Return size in bytes of (new) objects
//
//
{
	return existing_bytes + additional_bytes;
}

void Card_Objects::Object_To_Card(const Catalogue_Entry& cat_entry, const Object_Plan& object_plan)
//
//
// Copy object to flash card
//
//
{
	Object_Header	object_header;
	nval8		raw[OBJECT_HEADER_SIZE];

	The_Storage->Write_Object(cat_entry.Card_Address(), cat_entry.Populated_Bytes(), object_plan);

	object_header.signature = CRD_SIGN_OHD;

	object_header.catalogue_entry = &cat_entry;

	object_header.Export(raw);

	The_Storage->Write_Cached(cat_entry.Card_Address() - OBJECT_HEADER_SIZE, raw, OBJECT_HEADER_SIZE);

	dirty = true;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//
//
//
// Main
//
//
//
//
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void main_actual(int argument_count, char** argument_list)
{
	std::string		card_spec;
	std::list<Script_Line>	script_lines;
	int			argument_index;
	char*			argument_value;
	int			area_number = 1;
	bool			fake_format = false;
	bool			perform_list = false;
	bool			perform_format = false;
	bool			single_command = false;

	//
	// Announce
	//

	std::cout << "\n" CODE_NAME " " CODE_VERSION ", " CODE_COPYRIGHT << std::endl;

	//
	// Process arguments
	//

	argument_index = 0;

	while (++argument_index < argument_count)
	{
		argument_value = argument_list[argument_index];

		if (argument_value[0] == '-')
		{
			switch (argument_value[1])
			{
			case 'c' :
			case 'f' :
			case 'l' :
			case 's' :
#ifdef GOMMC
			case 'x' :
#endif
				{
					if (argument_value[2] != '\0')
						Issue_Usage_Error(MAKE_STRING("syntax: -" << argument_value[1] << ", used: " << argument_value));

					switch (argument_value[1])
					{
					case 'c' :
						single_command = true;
						break;
					case 'f' :
						perform_format = true;
						break;
					case 'l' :
						perform_list = true;
						break;
					case 'x' :
						fake_format = true;
						break;
					}
				}
				break;
#ifdef GOSDC
			case 'a' :
				{
					if (argument_value[2] == '\0')
						Issue_Usage_Error(MAKE_STRING("syntax: -a<number>, used: " << argument_value));

					if (strlen(&argument_value[2]) == 1)
						area_number = atoi(&argument_value[2]);
					else
						area_number = 0;
				}
				break;
#endif
			case 'd' :
				{
					if (argument_value[2] == '\0')
						Issue_Usage_Error(MAKE_STRING("syntax: -d" SYS_DOPTSYN ", used: " << argument_value));

					card_spec = &argument_value[2];
				}
				break;
			default :
				{
					Issue_Usage_Error(MAKE_STRING("unknown option: " << argument_value));
				}
				break;
			}
		}
		else
		{
			if (single_command)
				Add_Script_Line(script_lines, argument_value, true);
			else
				Add_Script_File(script_lines, argument_value);

			single_command = false;
		}
	}

	if (card_spec.size() == 0)
		Issue_Usage_Error("please specify the -d option");

	if (area_number < 1 || area_number > 99)
		Issue_Usage_Error("area number must be between 1 and 99 (inclusive)");

	if (script_lines.size() == 0)
		Issue_Usage_Error("no script lines to execute");

	//
	// Process
	//

	The_Area = (((OFFT_64) area_number) - 1) << 32;

	The_Storage = new Card_Storage(card_spec, CACHE_BUFFER_SIZE);

	The_Global_Header = new Card_Global_Header(CRD_ADDR_GHD, GLOBAL_HEADER_SIZE, perform_format && fake_format);

	The_Old_Catalogue = new Card_Catalogue(CRD_ADDR_CAT, The_Global_Header->Catalogue_Entries());

	if (perform_format)
	{
		std::cout << "\n!! Discarding old flash card contents, due to '-f' !!" << std::endl;

		The_Global_Header->Catalogue_Entries(0);

		The_Global_Header->Free_Data_Pointer(The_Old_Catalogue->Objects_End_Address());

		delete The_Old_Catalogue;

		The_Old_Catalogue = new Card_Catalogue(CRD_ADDR_CAT, The_Global_Header->Catalogue_Entries());
	}

	The_Old_Objects = new Card_Objects(0);

	std::cout << "\n- Reading commands" << std::endl;

	for (std::list<Script_Line>::iterator walker = script_lines.begin(); walker != script_lines.end(); walker++)
		The_Commands.Append(*walker);

	std::cout << "- Reading done" << std::endl;

	The_Commands.Evaluate();

	The_New_Catalogue = new Card_Catalogue(CRD_ADDR_CAT, The_Global_Header->Catalogue_Entries());

	The_New_Objects = new Card_Objects(The_Commands.Delta_Objects());

	The_Commands.Update();

	std::cout << "\n"
		"Flash card info + changes\n"
		"=========================\n"
		"Catalogue entries : " << The_Old_Catalogue->Size_In_Entries() << " (grows to " << The_New_Catalogue->Size_In_Entries() << ")"
		<< std::endl
		;

	std::cout << std::hex
		<< "Catalogue at      : " << std::setw(8) << The_Old_Catalogue->Base_Address() << " + " << std::setw(8) << The_Old_Catalogue->Size_In_Bytes()
		<< " (+ " << std::setw(8) << (The_New_Catalogue->Size_In_Bytes() - The_Old_Catalogue->Size_In_Bytes())
		<< " -> " << std::setw(8) << The_New_Catalogue->Base_Address() << " + " << std::setw(8) << The_New_Catalogue->Size_In_Bytes() << ")\n"
		<< "Objects at        : " << std::setw(8) << The_Old_Objects->Base_Address() << " + " << std::setw(8) << The_Old_Objects->Size_In_Bytes()
		<< " (+ " << std::setw(8) << (The_New_Objects->Size_In_Bytes() - The_Old_Objects->Size_In_Bytes())
		<< " -> " << std::setw(8) << The_New_Objects->Base_Address() << " + " << std::setw(8) << The_New_Objects->Size_In_Bytes() << ")\n"
		<< std::dec
		<< std::flush
		;

	if (The_New_Objects->Base_Address() < The_New_Catalogue->End_Address())
		Issue_Fatal_Error("the new objects do not fit");

	The_Commands.Execute();

	The_Global_Header->Catalogue_Entries(The_New_Catalogue->Size_In_Entries());

	The_Global_Header->Free_Data_Pointer(The_New_Objects->Base_Address());

	The_New_Objects->Flush();

	The_New_Catalogue->Flush();

	The_Global_Header->Flush();

	if (perform_list)
		The_New_Catalogue->List();

	//
	// Finish
	//

	std::cout << "\nFinished !\n";
}

void my_new_handler()
{
	throw std::bad_alloc();
}

int main(int argument_count, char** argument_list)
{
	std::cout << std::setfill('0');

	std::set_new_handler(&my_new_handler);

	if (sizeof(OFFT_64) < 8)
		Issue_Fatal_Error("no 64-bit file access");

	try
	{
		main_actual(argument_count, argument_list);
	}
	catch (std::bad_alloc&)
	{
		Issue_Fatal_Error("unexpectedly out of memory");
	}

	SYS_FINISH;

	return 0;
}


